package com.rocoinfo.utils.io;

import com.google.common.io.Files;
import com.rocoinfo.utils.base.Platform;
import com.rocoinfo.utils.text.StringUtil;
import org.apache.commons.lang3.StringUtils;

import java.util.HashMap;
import java.util.Map;

/**
 * 关于文件路径的工具集. 这个类只适合处理纯字符串的路径，如果是File对象或者Path对象的路径处理，建议直接使用Path类的方法。
 * 
 * @see {@link java.nio.file.Path}
 */
public class PathUtil {

	/**
	 * 传入queryString，返回去除pageNumber参数之后的部分。 例如：
	 * <ul>
	 * <li>"pageNumber=5&a=1&b=2" 结果 "a=1&b=2"</li>
	 * <li>"a=1&b=2&pageNumber=5" 结果 "a=1&b=2"</li>
	 * <li>null 结果 ""</li>
	 * </ul>
	 * @param queryString 未解码的查询字符串
	 * @return 去除pageNumber参数之后的部分
	 */
	public static String removeQueryStringPageNumber(String queryString) {
		if (StringUtils.isNotBlank(queryString)) {
			return queryString.replaceAll("(^|&)pageNumber=(\\d*?)(&|$)", "$1").replaceAll("&$", "");
		}
		return "";
	}

	/**
	 * 去掉url中的路径，留下请求参数部分
	 * @param strURL url地址
	 * @return url请求参数部分
	 */
	private static String resolveUrlQueryString(String strURL) {
		String strAllParam = null;
		String[] arrSplit;
		arrSplit = strURL.split("[?]");
		if (strURL.length() > 1) {
			if (arrSplit.length > 1) {
				if (arrSplit[1] != null) {
					strAllParam = arrSplit[1];
				}
			}
		}

		return strAllParam;
	}

	/**
	 * 解析出url参数中的键值对 如 "index.jsp?Action=del&id=123"，解析出Action:del,id:123存入map中
	 *
	 * @param url url地址
	 * @return url请求参数部分
	 */
	public static Map<String, String>  resolveQueryStringToMap(String url) {
		Map<String, String> mapRequest = new HashMap<>();

		String[] arrSplit;

		String strUrlParam = resolveUrlQueryString(url);
		if (strUrlParam == null) {
			return mapRequest;
		}
		arrSplit = strUrlParam.split("[&]");
		for (String strSplit : arrSplit) {
			String[] arrSplitEqual;
			arrSplitEqual = strSplit.split("[=]");

			// 解析出键值
			if (arrSplitEqual.length > 1) {
				// 正确解析
				mapRequest.put(arrSplitEqual[0], arrSplitEqual[1]);

			} else {
				if (!"".equals(arrSplitEqual[0])) {
					// 只有参数没有值，不加入
					mapRequest.put(arrSplitEqual[0], "");
				}
			}
		}
		return mapRequest;
	}

	/**
	 * 在Windows环境里，兼容Windows上的路径分割符，将 '/' 转回 '\'
	 */
	public static String normalizePath(String path) {
		if (Platform.FILE_PATH_SEPARATOR_CHAR == Platform.WINDOWS_FILE_PATH_SEPARATOR_CHAR
				&& StringUtils.indexOf(path, Platform.LINUX_FILE_PATH_SEPARATOR_CHAR) != -1) {
			return StringUtils.replaceChars(path, Platform.LINUX_FILE_PATH_SEPARATOR_CHAR,
					Platform.WINDOWS_FILE_PATH_SEPARATOR_CHAR);
		}
		return path;
	}



	/**
	 * 将路径整理，如 "a/../b"，整理成 "b"
	 * 去掉路径中最后的/，例如 F:/a/b/ => F:/a/b
	 */
	public static String simplifyPath(String path) {
		return Files.simplifyPath(path);
	}


	/**
	 * 以拼接路径名
	 */
	public static String concat(String baseName, String... appendName) {
		if (appendName.length == 0) {
			return baseName;
		}

		StringBuilder concatName = new StringBuilder();
		if (StringUtils.endsWith(baseName, String.valueOf(Platform.FILE_PATH_SEPARATOR_CHAR))) {
			concatName.append(baseName).append(appendName[0]);
		} else {
			concatName.append(baseName).append(Platform.FILE_PATH_SEPARATOR_CHAR).append(appendName[0]);
		}

		if (appendName.length > 1) {
			for (int i = 1; i < appendName.length; i++) {
				concatName.append(Platform.FILE_PATH_SEPARATOR_CHAR).append(appendName[i]);
			}
		}

		return concatName.toString();
	}

	/**
	 * 获得上层目录的路径
	 */
	public static String getParentPath(String path) {
		String parentPath = path;

		if (Platform.FILE_PATH_SEPARATOR.equals(parentPath)) {
			return parentPath;
		}

		parentPath = StringUtil.removeEnd(parentPath, Platform.FILE_PATH_SEPARATOR_CHAR);

		int idx = parentPath.lastIndexOf(Platform.FILE_PATH_SEPARATOR_CHAR);
		if (idx >= 0) {
			parentPath = parentPath.substring(0, idx + 1);
		} else {
			parentPath = Platform.FILE_PATH_SEPARATOR;
		}

		return parentPath;
	}

	/**
	 * 获得参数clazz所在的Jar文件的绝对路径
	 */
	public static String getJarPath(Class<?> clazz) {
		return clazz.getProtectionDomain().getCodeSource().getLocation().getFile();
	}



}
