package com.rocoinfo.utils.mapper;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.util.JSONPObject;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.*;

/**
 * <dl>
 * <dd>Description: json工具类，使用Jackson</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2016/11/1 17:25</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public class JsonUtil {
    public static final String JSON_TYPE = "application/json";

    //生成json时：只输出非null的属性
    public static final ObjectMapper mapper = JsonMapper.nonNullMapper().getMapper();

    private JsonUtil() {
    }


    /**
     * 将POJO转为Map
     */
    public static Map<String, Object> toMap(Object pojo) {
        if (pojo instanceof Map) {
            return (Map<String, Object>) pojo;
        } else {
            try {
                return PropertyUtils.describe(pojo);
            } catch (Exception e) {
                return null;
            }
        }
    }


    /**
     * 輸出JSONP格式
     */
    public String toJsonP(String functionName, Object obj) {
        return toJson(new JSONPObject(functionName, obj));
    }

    /**
     * 对象转成Json  如果对象为Null, 返回"null". 如果集合为空集合, 返回"[]".
     *
     * @param obj 对象支持: pojo，数组,Set<pojo>,List<pojo>,Map<Object,pojo>
     * @return
     */
    public static String toJson(Object obj) {
        if (obj == null) {
            return StringUtils.EMPTY;
        }
        String json;
        try {
            json = mapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            return StringUtils.EMPTY;
        }
        return json;
    }

    /**
     * Json串->对象 或简单Collection如List<String>
     * 如果JSON字符串为Null或"null"字符串, 返回Null.
     * 如果JSON字符串为"[]", 返回空集合.
     * 如需反序列化复杂Collection如List<MyBean>, 请使用fromJson(String, JavaType)
     * @param json  pojo实体类Json串
     * @param pojoClass 实体类class
     * @return
     */
    public static <T> T fromJson(String json, Class<T> pojoClass) {
        if (StringUtils.isBlank(json)) {
            return null;
        }

        try {
            return mapper.readValue(json, pojoClass);
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 将Json字符串转成Map
     *
     * @param json       json
     * @param keyClass   key的类型
     * @param valueClass value的类型
     * @param <K>        泛型Key
     * @param <V>        泛型Value
     * @return
     */
    public static <K, V> Map<K, V> fromJsonAsMap(String json, Class<K> keyClass, Class<V> valueClass) {
        if (StringUtils.isBlank(json)) {
            return new HashMap<>();
        }

        JavaType type = mapper.getTypeFactory().
                constructMapType(Map.class, keyClass, valueClass);
        try {
            Map<K, V> map = mapper.readValue(json, type);
            return map;
        } catch (IOException e) {
            return new HashMap<>();
        }
    }

    /**
     * Json串转成List,带泛型
     *
     * @param json  json串
     * @param pojoClass 实体类class
     * @param <T>   泛型
     * @return
     */
    public static <T> List<T> fromJsonAsList(String json, Class<T> pojoClass) {
        if (StringUtils.isBlank(json)) {
            return Collections.emptyList();
        }

        JavaType type = mapper.getTypeFactory().
                constructCollectionType(List.class, pojoClass);
        try {
            return mapper.readValue(json, type);
        } catch (IOException e) {
            return null;
        }
    }

    /**
    **
     * 反序列化复杂Collection如List<Bean>, 先使用createCollectionType()或contructMapType()构造类型, 然后调用本函数.
         * @see #createCollectionType(Class, Class...)
        */
    @SuppressWarnings("unchecked")
    public <T> T fromJson(String jsonString, JavaType javaType) {
        if (StringUtils.isEmpty(jsonString)) {
            return null;
        }

        try {
            return (T) mapper.readValue(jsonString, javaType);
        } catch (IOException e) {
            return null;
        }
    }


    /**
     * 构造Collection类型.
     */
    @SuppressWarnings("rawtypes")
    public JavaType contructCollectionType(Class<? extends Collection> collectionClass, Class<?> elementClass) {
        return mapper.getTypeFactory().constructCollectionType(collectionClass, elementClass);
    }

    /**
     * 构造Map类型.
     */
    @SuppressWarnings("rawtypes")
    public JavaType contructMapType(Class<? extends Map> mapClass, Class<?> keyClass, Class<?> valueClass) {
        return mapper.getTypeFactory().constructMapType(mapClass, keyClass, valueClass);
    }

}