package com.rocoinfo.utils.net;

import com.google.common.collect.Lists;
import com.rocoinfo.utils.base.ExceptionUtil;
import com.rocoinfo.utils.mapper.JsonUtil;
import com.rocoinfo.utils.text.Charsets;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;

public final class HttpUtil {

    private static Logger logger = LoggerFactory.getLogger(HttpUtil.class);


    /**
     * http get
     *
     * @param url
     * @param parameters
     * @return
     */
    public static String get(String url, final NameValuePair... parameters) {
        if (StringUtils.isBlank(url)) {
            throw new IllegalArgumentException("get请求地址不能为空");
        } else {
            String respTxt = StringUtils.EMPTY;
            HttpClient httpClient = createHttpClientInstance();
            try {
                if (null != parameters && parameters.length > 0) {
                    String encodedParams = encodeParameters(parameters);
                    if (-1 == url.indexOf("?")) {
                        url += "?" + encodedParams;
                    } else {
                        url += "&" + encodedParams;
                    }
                }
                HttpGet getMethod = new HttpGet(url);
                // 设置请求和传输超时时间
                RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(4000).setConnectTimeout(2000).build();
                getMethod.setConfig(requestConfig);
                HttpResponse resp = httpClient.execute(getMethod);
                if (resp.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                    respTxt = EntityUtils.toString(resp.getEntity(), Charsets.UTF_8_NAME);
                }
            } catch (Exception e) {
                e.printStackTrace();
                logger.error("httpGet请求异常：{},错误详细是：{}", url, ExceptionUtil.stackTraceText(e));
            } finally {
                HttpClientUtils.closeQuietly(httpClient);
            }
            return respTxt;
        }
    }

    public static String post(String postUrl, final NameValuePair... parameters) {
        return postWithResponseCharset(postUrl,null,parameters);
    }


    public static String postWithResponseCharset(String postUrl, String responseCharset, final NameValuePair... parameters) {
        HttpClient client = createHttpClientInstance();
        String respResult = null;
        try {
            HttpPost httPost = new HttpPost(postUrl);
            if (ArrayUtils.isNotEmpty(parameters)) {
                List<NameValuePair> params = Lists.newArrayList(parameters);
                httPost.setEntity(new UrlEncodedFormEntity(params, Charsets.UTF_8_NAME));
            }
            HttpResponse resp = client.execute(httPost);
            if (resp.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                HttpEntity entity = resp.getEntity();
                if(StringUtils.isBlank(responseCharset)) {
                    respResult = EntityUtils.toString(entity);
                }else {
                    respResult = EntityUtils.toString(entity, responseCharset);
                }
            }
        } catch (Exception e) {
            logger.error("http请求异常：url：{}，error：{}", postUrl, ExceptionUtil.stackTraceText(e));
        } finally {
            HttpClientUtils.closeQuietly(client);
        }
        return respResult;
    }

    /**
     * http post json
     * @param url
     * @param json
     * @param headers
     * @return
     */
    public static String postJson(String url, String json, Map<String, String> headers) {
        HttpClient client = createHttpClientInstance();
        HttpPost post = new HttpPost(url);
        String respResult = null;
        Integer statusCode = null;
        try {
            if (!MapUtils.isEmpty(headers)) {
                headers.keySet().forEach(key -> post.addHeader(key, headers.get(key)));
            }

            StringEntity stringEntity = new StringEntity(json, Charsets.UTF_8_NAME);
            stringEntity.setContentEncoding(Charsets.UTF_8_NAME);
            stringEntity.setContentType(JsonUtil.JSON_TYPE);
            post.setEntity(stringEntity);
            HttpResponse resp = client.execute(post);
            statusCode = resp.getStatusLine().getStatusCode();
            if (statusCode == HttpStatus.SC_OK) {
                HttpEntity entity = resp.getEntity();
                respResult = EntityUtils.toString(entity, Charsets.UTF_8_NAME);
            }
            logger.info("HttpUtils#post执行结束：{}, 状态码为：{}, 返回结果为：{}", url, statusCode, respResult);
        } catch (Exception ex) {
            logger.error("HttpUtils#post执行异常：{}, 状态码为：{}, 错误详细是：{}", url, statusCode, ex.getMessage());
        } finally {
            HttpClientUtils.closeQuietly(client);
        }
        return respResult;
    }

    /**
     * http put json
     *
     * @param url
     * @param json
     * @param headers
     * @return
     */
    public static String put(String url, String json, Map<String, String> headers) {
        HttpClient client = createHttpClientInstance();
        HttpPut put = new HttpPut(url);
        String respResult = null;
        Integer statusCode = null;
        try {
            if (!MapUtils.isEmpty(headers)) {
                headers.keySet().forEach(key -> put.addHeader(key, headers.get(key)));
            }
            StringEntity stringEntity = new StringEntity(json, Charsets.UTF_8_NAME);
            stringEntity.setContentEncoding(Charsets.UTF_8_NAME);
            stringEntity.setContentType(JsonUtil.JSON_TYPE);
            put.setEntity(stringEntity);

            HttpResponse resp = client.execute(put);
            statusCode = resp.getStatusLine().getStatusCode();
            if (statusCode == HttpStatus.SC_OK) {
                HttpEntity entity = resp.getEntity();
                respResult = EntityUtils.toString(entity, Charsets.UTF_8_NAME);
            }
            logger.info("HttpUtils#put执行结束：{}, 状态码为：{}, 返回结果为：{}", url, statusCode, respResult);
        } catch (Exception ex) {
            logger.error("HttpUtils#put执行异常：{}, 状态码为：{}, 错误详细是：{}", url, statusCode, ex.getMessage());
        } finally {
            HttpClientUtils.closeQuietly(client);
        }
        return respResult;
    }

    /**
     * 将NameValuePairs数组转变为字符串
     */
    private static String encodeParameters(final NameValuePair[] nameValues) {
        if (nameValues == null || nameValues.length == 0) {
            return StringUtils.EMPTY;
        }
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < nameValues.length; i++) {
            NameValuePair nameValue = nameValues[i];
            if (i == 0) {
                buffer.append(nameValue.getName() + "=" + nameValue.getValue());
            } else {
                buffer.append("&" + nameValue.getName() + "=" + nameValue.getValue());
            }
        }
        return buffer.toString();
    }

    public static HttpClient createHttpClientInstance() {
        HttpClientBuilder httpClientBuilder = HttpClientBuilder.create();
        RequestConfig.Builder reqConfigBuilder = RequestConfig.custom();
        reqConfigBuilder.setSocketTimeout(120000);
        reqConfigBuilder.setConnectTimeout(120000);
        reqConfigBuilder.setConnectionRequestTimeout(120000);
        RequestConfig reqConfig = reqConfigBuilder.build();

        SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(
                SSLContexts.createDefault(),
                SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
        httpClientBuilder.setSSLSocketFactory(sslsf);

        httpClientBuilder.setDefaultRequestConfig(reqConfig);
        return httpClientBuilder.build();
    }
}