package com.rocoinfo.utils.time;

import com.rocoinfo.utils.base.annotation.NotNull;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.commons.lang3.time.FastDateFormat;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;

/**
 * Date的parse()与format(), 采用Apache Common Lang中线程安全, 性能更佳的FastDateFormat
 * 
 * 注意Common Lang版本，3.5版才使用StringBuilder，3.4及以前使用StringBuffer.
 * 
 * 1. 常用格式的FastDateFormat定义, 常用格式直接使用这些FastDateFormat
 * 
 * 2. 日期格式不固定时的String<->Date 转换函数.
 * 
 * 3. 打印时间间隔，如"01:10:10"，以及用户友好的版本，比如"刚刚"，"10分钟前"
 * 
 * @see FastDateFormat#parse(String)
 * @see FastDateFormat#format(java.util.Date)
 * @see FastDateFormat#format(long)
 */
public class DateFormatUtil {

    /**
     * 定义常量
     **/
    public static final String YYYYMM = "yyyyMM";
    public static final String YYYY_MM = "yyyy-MM";
    public static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";
    public static final String YYYY_MM_DD = "yyyy-MM-dd";
    public static final String YYYYMMDD = "yyyyMMdd";
    public static final String YYYYMMDDHHMMSS = "yyyyMMddHHmmss";
    public static final String YYYY_MM_DD_HH_MM = "yyyy-MM-dd HH:mm";
    public static final String HOUR_MM_SS = "HHmmss";
    public static final String HOUR_MM_SS_TIME = "HH:mm:ss";
    public static final String TIME_ZONE = "GMT+08:00";
    public static final String START_TIME = " 00:00:00";//开始时间
    public static final String END_TIME = " 23:59:59"; //结束时间


    /**
     * 使用预设格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @return
     */
    public static Date parseToDateTime(String strDate) {
        return parseDate(strDate, YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 使用预设格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @return
     */
    public static Date parseToDate(String strDate) {
        return parseDate(strDate, YYYY_MM_DD);
    }


    /**
     * 将指定的日期转换成Unix时间戳
     *
     * @param date 需要转换的日期 yyyy-MM-dd HH:mm:ss
     * @return long 时间戳
     */
    public static long parseToMilliSecond(String date) {
        return  parseToMilliSecond(date,YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 将指定的日期转换成Unix时间戳
     *
     * @param date    需要转换的日期字符串
     * @param pattern 传入的指定日期字符串的格式
     * @return long 时间戳
     */
    public static long parseToMilliSecond(String date, String pattern) {
        Date dateTime = parseDate(date,pattern);
        if(dateTime == null) {
            return  0;
        }
        return  dateTime.getTime();
    }

    /**
     * 使用用户格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @param pattern 日期格式
     * @return
     */
    public static Date parseDate(String strDate, String pattern) {
        try {
            return FastDateFormat.getInstance(pattern).parse(strDate);
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }

    public static LocalDate parseToLocalDate(Date date) {
        if (date == null) {
            return null;
        }
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
    }

    public static LocalDateTime parseToLocalDateTime(Date date) {
        if (date == null) {
            return null;
        }
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
    }

    public static LocalDateTime parseToLocalDateTime(String dateStr, String pattern) {
        Date date = parseDate(dateStr, pattern);
        if (date == null) {
            return null;
        }
        return parseToLocalDateTime(date);
    }

    //// 日期格式化

    /**
     * 格式化成 YYYY_MM_DD
     * @param date
     */
    public static String formatToDate(Date date) {
        return formatDate(date,YYYY_MM_DD);
    }

    /**
     * 格式化成 YYYY_MM_DD_HH_MM_SS
     * @param date
     */
    public static String formatToDateTime(Date date) {
        return formatDate(date,YYYY_MM_DD_HH_MM_SS);
    }


    /**
     * 格式化成 HOUR_MM_SS
     * @param date
     */
    public static String formatToTime(Date date) {
        return formatDate(date,HOUR_MM_SS_TIME);
    }


    /**
     * 格式化成 YYYY_MM_DD_HH_MM_SS
     */
    public static String formatToDateTime(LocalDateTime dateTime) {
        return formatLocalDateTime(dateTime,YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 将指定的日期转换成指定格式的时间字符串
     *
     * @param dateTime 需要转换的日期
     * @param pattern  指定日期字符串格式
     * @return String 时间戳
     */
    public static String formatLocalDateTime(LocalDateTime dateTime, String pattern) {
        if(dateTime == null) {
            return StringUtils.EMPTY;
        }

        return dateTime.format(DateTimeFormatter.ofPattern(pattern));
    }



    /**
     * 根据字符串格式,日期字符串
     *
     * @param formatStr 格式化参数
     * @param dateStr   日期字符串
     * @return
     */
    public static String formatDate(String dateStr, String formatStr) {
        return new SimpleDateFormat(formatStr).format(parseToDate(dateStr));
    }

    /**
     * 将指定的日期转换成指定格式的时间字符串
     *
     * @param date    需要转换的日期
     * @param pattern 指定日期字符串格式
     * @return String 时间戳
     */
    public static String formatDate(Date date, @NotNull String pattern) {
        return FastDateFormat.getInstance(pattern).format(date);
    }

    /**
     * 格式化日期, 仅用于不固定pattern不固定的情况.
     *
     * 否否则直接使用本类中封装好的FastDateFormat.
     *
     * FastDateFormat.getInstance()已经做了缓存，不会每次创建对象，但直接使用对象仍然能减少在缓存中的查找.
     */
    public static String formatDate(long date,@NotNull String pattern) {
        return FastDateFormat.getInstance(pattern).format(date);
    }


    /**
     * 将LocalDate格式化成yyyy-MM-dd
     *
     * @param date localDate
     * @return 返回格式化后的字符串
     */
    public static String formatLocalDate(LocalDate date) {
        if (date == null) {
            return StringUtils.EMPTY;
        }
        return date.format(DateTimeFormatter.ofPattern(YYYY_MM_DD));
    }


    /////// 格式化间隔时间/////////
    /**
     * 按HH:mm:ss格式，格式化时间间隔
     *
     * endDate必须大于startDate，间隔可大于1天
     *
     * @see DurationFormatUtils
     */
    public static String formatDurationOnSecond(@NotNull Date startDate, @NotNull Date endDate) {
        return DurationFormatUtils.formatDuration(endDate.getTime() - startDate.getTime(), HOUR_MM_SS_TIME);
    }

    /**
     * 按HH:mm:ss格式，格式化时间间隔
     *
     * 单位为毫秒，必须大于0，可大于1天
     *
     * @see DurationFormatUtils
     */
    public static String formatDurationOnSecond(long durationMillis) {
        return DurationFormatUtils.formatDuration(durationMillis, HOUR_MM_SS_TIME);
    }
}