package com.rocoinfo.rocomall.entity.activity;

import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import org.springframework.data.annotation.Transient;
import org.springframework.util.CollectionUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.collect.Sets;
import com.rocoinfo.rocomall.entity.IdEntity;
import com.rocoinfo.rocomall.entity.Supplier;

/**
 * 活动
 * @author zhangmin
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class Activity extends IdEntity{

	private static final long serialVersionUID = 1177054672620656211L;

    private String name;

    private String promotionWord;

    private Type type;

    private Date startTime;
    private Date endTime;

    private Status status;
    
    private Supplier supplier;
    
    /**
     *参与此活动的商品，是否仅活动可以销售
     */
    private Boolean onlyActSale;
    
	/**
	 * 活动参与所需积分  
	 */
	private Integer joinCent;;
	
	/**
	 *竞拍增量区间（用户每次出价，必须大于最高出价）
	 */
	private Integer auctIncrMin;
	private Integer auctIncrMax;
	private Integer auctInitCent;
	private Integer auctTopN;
	
	private Integer seq;
	
	private Integer auctionMaxCent;
	
	@Transient
    private List<ActivitySku> activitySkus;

	public static enum Status {
		DRAFT("草稿"), //可以修改各项信息
		AUDIT("审批中"), READY_LIST("待上架"), REFUSE("已拒绝"), LIST(
			"已上架"), DELIST("已下架"), DISCARD("已作废");

		Status(String label) {
			this.label = label;
		}

		private String label;

		public String getLabel() {
			return label;
		}

		public static boolean isAllowChangeStatus(Status from, Status to) {
			EnumSet<Status> statusSets = getAllowChangeStatus(from);
			if (!CollectionUtils.isEmpty(statusSets)) {
				for (Status status : statusSets) {
					if (to == status) {
						return true;
					}
				}
			}
			return false;
		}

		/***
		 * 一个已有的活动可以有哪些允许状态
		 * 
		 * @param from
		 * @return
		 */
		public static EnumSet<Status> getAllowChangeStatus(Status from) {

			EnumSet<Status> statusSets = EnumSet.noneOf(Status.class);
			if (from == Status.DRAFT) {
				statusSets.add(Status.DISCARD);
				statusSets.add(Status.AUDIT);
			}
			if (from == Status.DISCARD) {
				// do nothing 
			}
			if (from == Status.AUDIT) {
				statusSets.add(Status.REFUSE);
				statusSets.add(Status.READY_LIST);
			}
			if (from == Status.REFUSE) {
				statusSets.add(Status.DISCARD);
				statusSets.add(Status.DRAFT);
			}
			if (from == Status.LIST) {
				statusSets.add(Status.DELIST);
			}
			if (from == Status.READY_LIST) {
				statusSets.add(Status.LIST);
				statusSets.add(Status.DISCARD);
			}
			if (from == Status.DELIST) {
				statusSets.add(Status.LIST);
				statusSets.add(Status.DISCARD);
			}
			return statusSets;
		}
	}

    public static enum Type {
    	AUCTION("拍卖"),
    	PRESALE("预售"),
        LIMITED_TIME_OFFER("限时抢购"),
        SCRAPE_CARD("刮刮卡"), 
    	TURN_TABLE("转盘抽奖"),
        SECKILL("秒杀");
        private String label;

        Type(String label) {
            this.label = label;
        }

        public String getLabel() {
            return label;
        }
    }
   
    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getPromotionWord() {
        return promotionWord;
    }

    public void setPromotionWord(String promotionWord) {
        this.promotionWord = promotionWord;
    }

    public Status getStatus() {
        return status;
    }

    public void setStatus(Status status) {
        this.status = status;
    }

    public Type getType() {
        return type;
    }

    public void setType(Type type) {
        this.type = type;
    }
    
    
    /**
     * 返回活动 可销售的库存数量,如果是套装或赠品，则取套装中sku最小库存
     */
    public int getAvailableStockQuantity(){
    	int availableStock = 0;
    	if(!CollectionUtils.isEmpty(activitySkus)){
    		Set<Integer> stockSet = Sets.newHashSet();
    		for(ActivitySku  actSku : activitySkus){
    			if(actSku!=null && actSku.getSku()!=null){
    				stockSet.add(actSku.getSku().getAvailableStockQuantity()/actSku.getCount());
    			}
    		}
    		if(!stockSet.isEmpty()){
    			availableStock = Collections.min(stockSet);
    		}
    	}
    	return availableStock;
    }
    
    
    /**
     * 返回活动价（积分数）
     */
    public int getActivitySaleCent(){
    	int actCent = 0;
    	if(!CollectionUtils.isEmpty(activitySkus)){
    		for(ActivitySku  actSku : activitySkus){
    			if(actSku!=null && actSku.getSku()!=null){
    				actCent += actSku.getCent();
    			}
    		}
    	}
    	
    	return actCent;
    }
    
    
    public Date getStartTime() {
		return startTime;
	}

	public void setStartTime(Date startTime) {
		this.startTime = startTime;
	}

	public Date getEndTime() {
		return endTime;
	}

	public void setEndTime(Date endTime) {
		this.endTime = endTime;
	}

	public Supplier getSupplier() {
		return supplier;
	}

	public void setSupplier(Supplier supplier) {
		this.supplier = supplier;
	}

	public Integer getJoinCent() {
		return joinCent;
	}

	public void setJoinCent(Integer joinCent) {
		this.joinCent = joinCent;
	}

	public Integer getAuctIncrMin() {
		return auctIncrMin;
	}

	public void setAuctIncrMin(Integer auctIncrMin) {
		this.auctIncrMin = auctIncrMin;
	}

	public Integer getAuctIncrMax() {
		return auctIncrMax;
	}

	public void setAuctIncrMax(Integer auctIncrMax) {
		this.auctIncrMax = auctIncrMax;
	}

	public Integer getAuctInitCent() {
		return auctInitCent;
	}

	public void setAuctInitCent(Integer auctInitCent) {
		this.auctInitCent = auctInitCent;
	}

	public Integer getAuctTopN() {
		return auctTopN;
	}

	public void setAuctTopN(Integer auctTopN) {
		this.auctTopN = auctTopN;
	}

	/**
	 * 返回活动是否开始
	 */
	public boolean isStart(){
		return this.startTime!=null && this.startTime.getTime() < System.currentTimeMillis();
	}
	
	/**
	 * 返回活动是否结束
	 */
	public boolean isStop(){
		return this.endTime!=null && this.endTime.getTime() < System.currentTimeMillis();
	}
	
	public Integer getSeq() {
		return seq;
	}

	public void setSeq(Integer seq) {
		this.seq = seq;
	}

	public List<ActivitySku> getActivitySkus() {
        return activitySkus;
    }

    public void setActivitySkus(List<ActivitySku> activitySkus) {
        this.activitySkus = activitySkus;
    }
    
	
	public Boolean getOnlyActSale() {
		return onlyActSale;
	}

	public void setOnlyActSale(Boolean onlyActSale) {
		this.onlyActSale = onlyActSale;
	}

	/**
	 * 能否参加活动
	 */
	public boolean isCanJoin(){
		return this.status == Status.LIST && isStart() && !isStop() && startTime.before(endTime);
	}
	
	/***
	 * 活动结束时间，因为网络延迟添加补偿值
	 * @param compensation
	 * @return
	 */
	public boolean isCanJoinAfterAddcompensation(){
		if(isCanJoin()){
			return true;
		}
		int compensation = 3000;
		if(this.status == Status.LIST && isStart()){
			return (this.getEndTime().getTime()+compensation>=System.currentTimeMillis());
		}
		return false;
	}

	public Integer getAuctionMaxCent() {
		return auctionMaxCent;
	}

	public void setAuctionMaxCent(Integer auctionMaxCent) {
		this.auctionMaxCent = auctionMaxCent;
	}
}