package com.rocoinfo.rocomall.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springside.modules.utils.Collections3;

import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.Supplier;
import com.rocoinfo.rocomall.entity.account.AdminRole;
import com.rocoinfo.rocomall.entity.account.AdminSupplier;
import com.rocoinfo.rocomall.entity.account.AdminUser;
import com.rocoinfo.rocomall.entity.account.Role;
import com.rocoinfo.rocomall.entity.account.RolePermission;
import com.rocoinfo.rocomall.enumconst.SwitchStatus;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.repository.account.AdminRoleDao;
import com.rocoinfo.rocomall.repository.account.AdminSupplierDao;
import com.rocoinfo.rocomall.repository.account.AdminUserDao;
import com.rocoinfo.rocomall.repository.account.RoleDao;
import com.rocoinfo.rocomall.repository.account.RolePermissionDao;
import com.rocoinfo.rocomall.service.IAdminUserService;
import com.rocoinfo.rocomall.service.impl.product.SupplierService;
import com.rocoinfo.rocomall.utils.PasswordUtil;

/**
 * <dl>
 * <dd>描述: 管理员用户Service</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2015年8月3日 上午10:05:36</dd>
 * <dd>创建人： 张敏</dd>
 * </dl>
 */
@Service
public class AdminUserService extends CrudService<AdminUserDao, AdminUser> implements IAdminUserService {

	@Autowired
	private AdminRoleDao adminRoleDao;
	@Autowired
	private RolePermissionDao rolePermissionDao;
	@Autowired
	private AdminSupplierDao adminSupplierDao;
	@Autowired
	private RoleDao roleDao;
	@Autowired
	private SupplierService supplierService;

	/**
	 * 更新登录密码
	 */
	@Override
	public void updateLoginPassword(final long adminId, final String newPlainPwd) {
		if (adminId > 0) {
			AdminUser admin = new AdminUser();
			admin.setId(adminId);
			admin.setPlainPassword(newPlainPwd);
			PasswordUtil.entryptPassword(admin);
			this.update(admin);
		}
	}

	/**
	 * 校验登录密码是否正确
	 * 
	 * @param username 登录名
	 * @param loginPassword 明文登录密码
	 */
	@Override
	public boolean isLoginPasswordCorrect(final String username, final String loginPassword) {
		AdminUser adm = this.entityDao.getByUsername(username);
		if (adm == null || adm.getPassword() == null || adm.getSalt() == null || loginPassword == null) {
			return false;
		}
		return PasswordUtil.hashPassword(loginPassword, adm.getSalt()).equals(adm.getPassword());
	}

	/**
	 * 判断用户名是否已经被注册
	 * 
	 * @param username 登录用户名
	 */
	@Override
	public boolean isUsernameRegisted(String username) {
		return this.entityDao.getByUsername(username) != null;
	}

	/**
	 * 保存管理员，并保存相关的角色和能够管理员的供应商
	 * 
	 * @param admin throws ServiceException 如果admin用户名已存在，则抛出此异常
	 */
	@Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
	public void saveAdminAndRole(final AdminUser admin) {
		removeNullRoles(admin);
		removeNullSuppliers(admin);
		if (isUsernameRegisted(admin.getUsername())) {
			throw new ServiceException("管理员已存在: " + admin.getUsername());
		}

		if (StringUtils.isNotEmpty(admin.getPlainPassword())) {
			PasswordUtil.entryptPassword(admin);
		}

		admin.setStatus(SwitchStatus.OPEN);
		insert(admin);

		if (CollectionUtils.isNotEmpty(admin.getRoles())) {
			for (Role role : admin.getRoles()) {
				adminRoleDao.insert(new AdminRole(admin.getId(), role.getId()));
			}
		}

		//添加能够管理的供应商
		if (CollectionUtils.isNotEmpty(admin.getSuppliers())) {
			for (Supplier supplier : admin.getSuppliers()) {
				adminSupplierDao.insert(new AdminSupplier(admin.getId(), supplier.getId()));
			}
		}
	}

	public void updateAdminAndRole(AdminUser admin) {
		removeNullRoles(admin);
		removeNullSuppliers(admin);
		if (StringUtils.isNotEmpty(admin.getPlainPassword())) {
			PasswordUtil.entryptPassword(admin);
		}

		//更新角色
		adminRoleDao.deleteByAdminId(admin.getId());
		if (CollectionUtils.isNotEmpty(admin.getRoles())) {
			for (Role role : admin.getRoles()) {
				adminRoleDao.insert(new AdminRole(admin.getId(), role.getId()));
			}
		}

		//更新供应商
		adminSupplierDao.deleteByAdminId(admin.getId());
		if (CollectionUtils.isNotEmpty(admin.getSuppliers())) {
			for (Supplier supplier : admin.getSuppliers()) {
				adminSupplierDao.insert(new AdminSupplier(admin.getId(), supplier.getId()));
			}
		}
		update(admin);
	}

	public AdminUser getAdminByUsername(String username) {
		return this.entityDao.getByUsername(username);
	}

	public Role getRoleById(final Long roleId) {
		for (Role role : getProxy().findAllRole()) {
			if (role.getId().equals(roleId)) {
				return role;
			}
		}
		return null;
	}

	public void buildRolePermissionDetail(Role role) {
		List<RolePermission> rolePermissions = rolePermissionDao.findByRoleId(role.getId());
		List<String> permissions = new ArrayList<String>(rolePermissions.size());
		for (RolePermission rolePermission : rolePermissions) {
			permissions.add(rolePermission.getPermission());
		}
		role.setPermissions(permissions);
	}

	public void buildAdminPermissionDetail(AdminUser admin) {
		List<Long> roleIds = adminRoleDao.findRoleIdsByAdminId(admin.getId());
		List<Role> roles = new ArrayList<Role>(roleIds.size());
		for (Long roleId : roleIds) {
			Role role = getRoleById(roleId);
			buildRolePermissionDetail(role);
			roles.add(role);
		}
		admin.setRoles(roles);

		// 供应商集合
		List<Supplier> suppliers = supplierService.findUseableSupplysByAdminId(admin.getId());
		if (suppliers == null) {
			suppliers = Collections.emptyList();
		}
		admin.setSuppliers(suppliers);
	}

	//@Cacheable(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ROLE_ALL_KEY)
	public List<Role> findAllRole() {
		return roleDao.findAll();
	}

	@CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ROLE_ALL_KEY)
	@Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
	public void saveRoleAndPermissions(Role role) {
		removeNullRoles(role);
		if (role != null && StringUtils.isNotBlank(role.getName())) {
			if (roleDao.countName(role) > 0) {
				throw new IllegalArgumentException("角色名称已经存在");
			}
		}

		roleDao.insert(role);
		if (CollectionUtils.isNotEmpty(role.getPermissions())) {
			for (String perm : role.getPermissions()) {
				rolePermissionDao.insert(new RolePermission(role.getId(), perm));
			}
		}
	}

	@CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ROLE_ALL_KEY)
	@Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
	public void updateRoleAndPermissions(Role role) {
		removeNullRoles(role);
		if (roleDao.countName(role) > 0) {
			throw new IllegalArgumentException("角色名称已经存在");
		}
		roleDao.update(role);
		rolePermissionDao.deleteByRoleId(role.getId());
		if (CollectionUtils.isNotEmpty(role.getPermissions())) {
			for (String perm : role.getPermissions()) {
				rolePermissionDao.insert(new RolePermission(role.getId(), perm));
			}
		}
	}

	@Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
	@CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ROLE_ALL_KEY)
	public void deleteRoleAndPermissionByRoleId(final Long roleId) {
		adminRoleDao.deleteByRoleId(roleId);
		rolePermissionDao.deleteByRoleId(roleId);
		roleDao.deleteById(roleId);
	}

	public void switchAdminStatus(Long userId, SwitchStatus status) {
		AdminUser admin = new AdminUser();
		admin.setId(userId);
		admin.setStatus(status);
		this.update(admin);
	}

	private void removeNullRoles(AdminUser user) {
		if (user != null && Collections3.isNotEmpty(user.getRoles())) {

			for (Iterator<Role> iterator = user.getRoles().iterator(); iterator.hasNext();) {
				Role role = iterator.next();
				if (role == null || role.getId() == null) {
					iterator.remove();
				}
			}
		}
	}

	private void removeNullSuppliers(AdminUser user) {
		if (user != null && Collections3.isNotEmpty(user.getSuppliers())) {

			for (Iterator<Supplier> iterator = user.getSuppliers().iterator(); iterator.hasNext();) {
				Supplier supplier = iterator.next();
				if (supplier == null || supplier.getId() == null) {
					iterator.remove();
				}
			}
		}
	}

	/**
	 * 剔除空的元素
	 * 
	 * @param role
	 */
	private void removeNullRoles(Role role) {
		if (role != null && Collections3.isNotEmpty(role.getPermissions())) {
			for (Iterator<String> iterator = role.getPermissions().iterator(); iterator.hasNext();) {
				String permission = iterator.next();
				if (StringUtils.isBlank(permission)) {
					iterator.remove();
				}
			}
		}
	}

	public List<AdminUser> findAllWithStatus(SwitchStatus status) {
		List<AdminUser> all = findAll();
		if (status == null) {
			return all;
		}

		List<AdminUser> opens = new ArrayList<AdminUser>();
		List<AdminUser> locks = new ArrayList<AdminUser>();
		for (AdminUser advType : all) {
			if (advType.getStatus() == SwitchStatus.OPEN) {
				opens.add(advType);
			} else {
				locks.add(advType);
			}
		}
		if (status == SwitchStatus.OPEN) {
			return opens;
		} else {
			return locks;
		}
	}

	public AdminUserService getProxy() {
		return (AdminUserService) AopContext.currentProxy();
	}

}
