package com.rocoinfo.rocomall.service.impl.order;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.entity.order.InstoreOrder;
import com.rocoinfo.rocomall.entity.order.InstoreOrder.ReceiveStatus;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.entity.workorder.WorkOrderRefundDetail;
import com.rocoinfo.rocomall.repository.order.InstoreOrderDao;
import com.rocoinfo.rocomall.service.order.IInstoreOrderService;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.product.ISkuService;
import com.rocoinfo.rocomall.service.workorder.IWorkOrderRefundDetailService;

/**
 * <dl>
 * <dd>描述:入库单Service实现</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>@创建时间：2015/11/12 17:50</dd>
 * <dd>@author：huyt</dd>
 * </dl>
 */
@Service
public class InstoreOrderService extends CrudService<InstoreOrderDao, InstoreOrder> implements IInstoreOrderService {

	@Autowired
	private IOrderItemService orderItemService;
	@Autowired
	private IWorkOrderRefundDetailService refundDetailService;
	@Autowired
	private ISkuService skuService;

	@Transactional
	@Override
	public void insert(InstoreOrder instoreOrder) {
		if (instoreOrder != null) {
			super.insert(instoreOrder);
			if (ReceiveStatus.FINISHED.equals(instoreOrder.getReceiveState())) {
				skuService.incrOrDecrStock(instoreOrder.getSku().getId(), instoreOrder.getQuantity(), instoreOrder.getWarehouse().getId());
			}
		}
	}

	/**
	 * 搜索商品的入库记录
	 * 
	 * @param productId 产品Id
	 */
	public Page<InstoreOrder> searchInstoreRecordsByProdId(Long productId, Pageable pageable) {
		List<InstoreOrder> pageData = Collections.emptyList();
		Long count = 0L;
		if (productId != null && productId > 0) {
			Map<String, Object> params = Maps.newHashMap();
			params.put("productId", productId);
			params.put(Constants.PAGE_OFFSET, pageable.getPageNumber());
			params.put(Constants.PAGE_SIZE, pageable.getPageSize());
			params.put(Constants.PAGE_SORT, pageable.getSort());

			count = this.entityDao.searchTotalInstoreRecordsByProdId(params);
			if (count > 0) {
				pageData = entityDao.searchInstoreRecordsByProdId(params);
			}
		}
		return new PageImpl<InstoreOrder>(pageData, pageable, count);
	}

	//退货入库单,确认收货
	@Transactional
	public void refundConfirmReceive(InstoreOrder instoreOrder) {
		if (ReceiveStatus.UNFINISHED.equals(instoreOrder.getReceiveState())) {
			InstoreOrder inStoreUpdate = new InstoreOrder();
			inStoreUpdate.setId(instoreOrder.getId());
			inStoreUpdate.setReceiveState(ReceiveStatus.FINISHED);
			inStoreUpdate.setReceiveDateTime(new Date());
			this.entityDao.update(inStoreUpdate);

			WorkOrderRefundDetail refundDetailItem = new WorkOrderRefundDetail();
			refundDetailItem.setRefunded(Boolean.TRUE);
			refundDetailItem.setId(instoreOrder.getRefundDetailId());
			refundDetailService.update(refundDetailItem);

			//退回库存
			this.skuService.incrOrDecrStock(instoreOrder.getSku().getId(), instoreOrder.getQuantity(), null);

			//更新订单状态
			OrderItem updateItem = new OrderItem();
			updateItem.setAdmStatus(OrderItem.AdmStatus.REFUND_RECEIVED);
			updateItem.setStatus(OrderItem.Status.REFUNDING);
			updateItem.setOrderCode(instoreOrder.getOrderCode());
			orderItemService.update(updateItem);
		}
	}
}
