package com.rocoinfo.rocomall.entity.wx;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import com.rocoinfo.rocomall.enumconst.WXMsgType;
import com.rocoinfo.rocomall.utils.CDATAConverter;
import com.rocoinfo.rocomall.utils.CodeGenerator;
import com.rocoinfo.rocomall.utils.XStreamUtils;
import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamConverter;

/**
 * 微信服务器推送过来的微信消息/自动回复的响应消息 微信消息： 接收消息：xml格式
 */
@XStreamAlias("xml")
@XStreamConverter(CDATAConverter.class)
public class WXMsg extends AbstractWX implements Serializable {

	/**
	 */
	private static final long serialVersionUID = -7110176973111728271L;

	/**
	 * 消息id，64位整型（接受到的消息的id,由微信生成）
	 */
	@XStreamAlias("MsgId")
	private Long msgId;

	/**
	 * 对于接收到的消息： 开发者微信号
	 */
	@XStreamAlias("ToUserName")
	private String toUserName;

	/**
	 * 发送方帐号（一个OpenID）
	 */
	@XStreamAlias("FromUserName")
	private String fromUserName;

	/**
	 * 消息创建时间 （单位:秒） 例如：1348831860
	 */
	@XStreamAlias("CreateTime")
	private Integer createTime;

	/**
	 * 消息类型{文本消息：text, 图片消息:image,语音消息：voice，视频消息：video,链接消息:link,事件推送:event}
	 */
	@XStreamAlias("MsgType")
	private String msgType;

	@XStreamAlias("Recognition")
	private String recognition;

	/**
	 * 接收事件推送 事件类型 subscribe(订阅)、unsubscribe(取消订阅) 自定义菜单：CLICK
	 */
	@XStreamAlias("Event")
	private String event;

	/**
	 * 事件KEY值，与自定义菜单接口中KEY值对应
	 */
	@XStreamAlias("EventKey")
	private String eventKey;

	/**
	 * 消息内容
	 */
	@XStreamAlias("Content")
	private String content;

	/**
	 * 图片消息的图片链接
	 */
	@XStreamAlias("PicUrl")
	private String picUrl;

	/**
	 * 媒体ID 图片消息媒体id/语音消息媒体id/视频消息媒体id，可以调用多媒体文件下载接口拉取数据。
	 */
	@XStreamAlias("MediaId")
	private String mediaId;

	/**
	 * 语音消息的语音格式 如amr，speex等
	 */
	@XStreamAlias("Format")
	private String format;

	/**
	 * 视频消息缩略图的媒体id，可以调用多媒体文件下载接口拉取数据
	 */
	@XStreamAlias("ThumbMediaId")
	private String humbMediaId;

	/**
	 * 地理位置纬度 23.134521
	 */
	@XStreamAlias("Location_X")
	private BigDecimal locationX;

	/**
	 * 地理位置经度
	 */
	@XStreamAlias("Location_Y")
	private BigDecimal locationY;

	/**
	 * 地图缩放大小
	 */
	@XStreamAlias("Scale")
	private BigDecimal scale;

	/**
	 * 地理位置信息
	 */
	@XStreamAlias("Label")
	private String label;

	/**
	 * 链接消息的 消息标题
	 */
	@XStreamAlias("Title")
	private String title;

	/**
	 * 链接消息的 消息描述
	 */
	@XStreamAlias("Description")
	private String description;

	/**
	 * 链接消息 链接url
	 */
	@XStreamAlias("Url")
	private String url;

	/**
	 * 图文消息个数 限制为10条以内
	 */
	@XStreamAlias("ArticleCount")
	private Integer articleCount;

	/**
	 * 自动回复图文消息体 多条图文消息信息，默认第一个item为大图,注意，如果图文数超过10，则将会无响应
	 */
	@XStreamAlias("Articles")
	private List<WXItem> articles;

	/**
	 * 自动回复的 音乐消息体
	 */
	@XStreamAlias("Music")
	private WXMusic music;

	/**
	 * 封装文本消息内容
	 */
	private void populateTxtMsgContent(MsgEntity entity) {
		entity.setContent(getContent());
	}

	/**
	 * 封装 图片消息内容
	 * 
	 * @param entity
	 */
	private void populateImageMsgContent(MsgEntity entity) {
		entity.setPicUrl(getPicUrl());
		entity.setMediaId(getMediaId());
	}

	/**
	 * 封装 语音消息内容
	 * 
	 * @param entity
	 */
	private void populateVoiceMsgContent(MsgEntity entity) {
		entity.setMediaId(getMediaId());
		entity.setFormat(getFormat());
	}

	/**
	 * 封装 视频消息内容
	 * 
	 * @param entity
	 */
	private void populateVideoMsgContent(MsgEntity entity) {
		entity.setMediaId(getMediaId());
		entity.setThumbMediaId(getHumbMediaId());
	}

	/**
	 * 封装 链接消息 的内容
	 */
	private void populateLinkMsgContent(MsgEntity entity) {
		entity.setUrl(getUrl());
		entity.setTitle(getTitle());
		entity.setDescription(getDescription());
	}

	/**
	 * 封装 地理位置消息 的内容
	 * 
	 * @param entity
	 */
	private void populateLocationMsgContent(MsgEntity entity) {
		entity.setLocationX(getLocationX());
		entity.setLocationY(getLocationY());
		entity.setScale(getScale());
		entity.setLabel(getLabel());
	}

	//保存上行的消息时 调用
	@SuppressWarnings("serial")
	public SingleMsgSingleReceiver valueOfWxMessage() {

		SingleMsgSingleReceiver txtMessage = new SingleMsgSingleReceiver(CodeGenerator.generateWxMessageId()) {

			@Override
			public MsgEntity getMsgEntity() {
				MsgEntity entity = new MsgEntity();
				entity.setMsgType(WXMsgType.valueOf(getMsgType()));
				entity.setMessageId(getMessageId());
				entity.setCreateTime((int) (long) getCreateTime());
				entity.setMsgId(getMsgId());
				switch (entity.getMsgType()) {
				case text:
					populateTxtMsgContent(entity);
					break;

				case image:
					populateImageMsgContent(entity);
					break;

				case voice:
					populateVoiceMsgContent(entity);
					break;

				case video:
					populateVideoMsgContent(entity);
					break;

				case link:
					populateLinkMsgContent(entity);
					break;
				case location:
					populateLocationMsgContent(entity);
					break;
				}
				return entity;
			}

			@Override
			public MsgReceiverEntity getMsgReceiver() {
				MsgReceiverEntity receiver = new MsgReceiverEntity();
				receiver.setDateline((int) (long) getCreateTime());
				receiver.setMessageId(getMessageId());
				receiver.setReceiver(getToUserName());
				receiver.setSender(getFromUserName());
				return receiver;
			}
		};
		return txtMessage;
	}

	public String getToUserName() {
		return toUserName;
	}

	public void setToUserName(String toUserName) {
		this.toUserName = toUserName;
	}

	public String getFromUserName() {
		return fromUserName;
	}

	public void setFromUserName(String fromUserName) {
		this.fromUserName = fromUserName;
	}

	public Integer getCreateTime() {
		return createTime;
	}

	public void setCreateTime(Integer createTime) {
		this.createTime = createTime;
	}

	public String getMsgType() {
		return msgType;
	}

	public void setMsgType(String msgType) {
		this.msgType = msgType;
	}

	public String getEvent() {
		return event;
	}

	public void setEvent(String event) {
		this.event = event;
	}

	public String getEventKey() {
		return eventKey;
	}

	public void setEventKey(String eventKey) {
		this.eventKey = eventKey;
	}

	public String getContent() {
		return content;
	}

	public void setContent(String content) {
		this.content = content;
	}

	public String getPicUrl() {
		return picUrl;
	}

	public void setPicUrl(String picUrl) {
		this.picUrl = picUrl;
	}

	public String getMediaId() {
		return mediaId;
	}

	public void setMediaId(String mediaId) {
		this.mediaId = mediaId;
	}

	public String getFormat() {
		return format;
	}

	public void setFormat(String format) {
		this.format = format;
	}

	public String getHumbMediaId() {
		return humbMediaId;
	}

	public void setHumbMediaId(String humbMediaId) {
		this.humbMediaId = humbMediaId;
	}

	public BigDecimal getLocationX() {
		return locationX;
	}

	public void setLocationX(BigDecimal locationX) {
		this.locationX = locationX;
	}

	public BigDecimal getLocationY() {
		return locationY;
	}

	public void setLocationY(BigDecimal locationY) {
		this.locationY = locationY;
	}

	public String getLabel() {
		return label;
	}

	public BigDecimal getScale() {
		return scale;
	}

	public void setScale(BigDecimal scale) {
		this.scale = scale;
	}

	public void setLabel(String label) {
		this.label = label;
	}

	public String getTitle() {
		return title;
	}

	public void setTitle(String title) {
		this.title = title;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getUrl() {
		return url;
	}

	public void setUrl(String url) {
		this.url = url;
	}

	public Integer getArticleCount() {
		return articleCount;
	}

	public void setArticleCount(Integer articleCount) {
		this.articleCount = articleCount;
	}

	public List<WXItem> getArticles() {
		return articles;
	}

	public void setArticles(List<WXItem> articles) {
		this.articles = articles;
	}

	public Long getMsgId() {
		return msgId;
	}

	public void setMsgId(Long msgId) {
		this.msgId = msgId;
	}

	public WXMusic getMusic() {
		return music;
	}

	public void setMusic(WXMusic music) {
		this.music = music;
	}

	public static void main(String[] args) {
		WXMsg wxMsg = new WXMsg();
		wxMsg.setToUserName("其实小小鱼");
		wxMsg.setCreateTime((int) (System.currentTimeMillis() / 1000L));
		wxMsg.setMsgType("news");
		wxMsg.setContent("今天的天气情况");
		wxMsg.setArticleCount(2);
		WXItem item1 = new WXItem();
		item1.setTitle("北京天气");
		item1.setDescription("北京天气情况");
		item1.setPicurl("http://localhost:8080/pic/beijing_raining.jpg");
		item1.setUrl("http://localhost:8080/link/beijing.htm");

		WXItem item2 = new WXItem();
		item2.setTitle("上海天气");
		item2.setDescription("上海天气情况");
		item2.setPicurl("http://localhost:8080/pic/shanghai_raining.jpg");
		item2.setUrl("http://localhost:8080/link/shanghai.htm");

		List<WXItem> itemList = new ArrayList<WXItem>();

		itemList.add(item1);
		itemList.add(item2);
		wxMsg.setArticles(itemList);

		XStream xStream = XStreamUtils.getDefaultXStream();
		System.out.println(xStream.toXML(wxMsg));
		wxMsg = (WXMsg) xStream.fromXML(xStream.toXML(wxMsg));
		System.out.println(wxMsg.getMsgType());
		System.out.println(wxMsg.getLocationX());
	}

	public String getRecognition() {
		return recognition;
	}

	public void setRecognition(String recognition) {
		this.recognition = recognition;
	}
}
