package com.rocoinfo.rocomall.pay;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.BankPaylog;
import com.rocoinfo.rocomall.entity.order.Order;
import com.rocoinfo.rocomall.entity.order.OrderItem.AdmStatus;
import com.rocoinfo.rocomall.entity.order.OrderItem.Status;
import com.rocoinfo.rocomall.enumconst.BankType;
import com.rocoinfo.rocomall.enumconst.PayStatus;
import com.rocoinfo.rocomall.pay.alipay.AlipayInterfaceHelper;
import com.rocoinfo.rocomall.pay.alipay.AlipayInterfaceHelper.AlipayQueryBaseMeta;
import com.rocoinfo.rocomall.pay.chinabank.ChinaBankPayHelper;
import com.rocoinfo.rocomall.pay.chinabank.ChinaBankPayHelper.ChBankQueryBaseMeta;
import com.rocoinfo.rocomall.pay.chinapay.ChinaPayInterfaceHelper;
import com.rocoinfo.rocomall.pay.kuaiqian.KuaiQianParm;
import com.rocoinfo.rocomall.pay.kuaiqian.KuaiQianPayHelper;
import com.rocoinfo.rocomall.pay.wechat.NotifyResultParm;
import com.rocoinfo.rocomall.pay.wechat.UnifiedOrderParm;
import com.rocoinfo.rocomall.pay.wechat.WxpayHelper;
import com.rocoinfo.rocomall.repository.BankPayLogDao;
import com.rocoinfo.rocomall.service.order.IOrderItemService;
import com.rocoinfo.rocomall.service.order.IOrderService;
import com.rocoinfo.rocomall.utils.XmlUtil;

/**
 * 支付日志
 * <dl>
 * <dd>描述:</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>@创建时间：2015-11-6 下午2:25:43</dd>
 * <dd>@author： 张文山</dd>
 * </dl>
 */
@Service
public class BankPayLogService extends CrudService<BankPayLogDao, BankPaylog> implements IBankPayLogService {
	
	@Autowired
	private IOrderService orderService;
	@Autowired
	private IOrderItemService orderItemService;

	@Override
	public CommonPayDto buildPayInfo(Long orderId, String returnUrl, String notifyUrl, BankType bankType) {
		Order order = orderService.getById(orderId);
		if(order == null){
			throw new ServiceException("不存在订单");
		}
		// 支付金额 单位元
		double amount = order.getCashAmt()+order.getDeliverFee();
		String productName = "电脑";// 商品名称
		Date orderCreateTm = new Date();// 订单创建时间
		// TODO 以上三个值从订单中获取
		if (isPaySuccess(orderId)) {
			throw new ServiceException("该订单已经支付成功");
		}
		BankPaylog log = createOrFindBankPayLogByOrderId(orderId, bankType, amount, orderCreateTm);
		Object payParm = null;// 返回的map对象
		String url = null;// 页面跳转url,有的支付不需要此值
		String methodType = "post";
		switch (bankType) {
		case ALIPAY:
			payParm = AlipayInterfaceHelper.buildQueryParam(new AlipayQueryBaseMeta(log.getBankOrderId(), productName, amount, returnUrl, notifyUrl));
			url = AlipayInterfaceHelper.ALIPAY_GATEWAY_NEW;
			methodType = "get";
			break;
		case CHINAPAY:
			payParm = ChinaPayInterfaceHelper.buildQueryParam(log.getBankOrderId(), amount, notifyUrl, returnUrl);
			url = ChinaPayInterfaceHelper.URL;
			break;
		case CHINABANK:
			ChBankQueryBaseMeta chbank = new ChBankQueryBaseMeta(log.getBankOrderId(), amount, returnUrl, notifyUrl);
			payParm = ChinaBankPayHelper.buildQueryParam(chbank);
			url = ChinaBankPayHelper.CHINABANK_GATEWAY_NEW;
			break;
		case WECHAT:
			int am = (int) (amount * 100);
			payParm = WxpayHelper.wxUnifiedorder(new UnifiedOrderParm(productName, log.getBankOrderId(), am, notifyUrl));
			break;
		case KUAIQIAN:
			int amk = (int) (amount * 100);
			payParm = KuaiQianPayHelper.buildQueryParam(new KuaiQianParm(log.getBankOrderId(), String.valueOf(amk), notifyUrl));
			url = KuaiQianPayHelper.URL;
			break;
		}
		return new CommonPayDto(bankType, url, payParm, methodType);
	}

	private BankPaylog createOrFindBankPayLogByOrderId(Long orderId, BankType bankType, double amount, Date orderCreateTm) {
		String orderidStr = String.valueOf(orderId);
		BankPaylog log = findByOrderidAndType(orderidStr, bankType);
		if (log != null) {
			return log;
		}
		SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmssS");
		String timStr = formatter.format(orderCreateTm);// 17位字符串
		String tempOrderId = timStr + orderidStr;
		switch (bankType) {
		case ALIPAY:
			tempOrderId = StringUtils.substring(tempOrderId, tempOrderId.length() - 12);
			// 110022334035
			break;
		case CHINAPAY:
			// 银联支付订单 1598523685147589
			tempOrderId = StringUtils.substring(tempOrderId, tempOrderId.length() - 16);
			break;
		case CHINABANK:
			SimpleDateFormat format = new SimpleDateFormat(Constants.YYYYMMDD_NJOIN);
			tempOrderId = format.format(orderCreateTm) + "-" + ChinaBankPayHelper.v_mid + "-" + orderidStr;
			// 订单编号标准格式为：订单生成日期(yyyymmdd)-商户编号-商户流水号。订单编号所有字符总和不可超过64位。
			break;
		case WECHAT:
			tempOrderId = StringUtils.substring(tempOrderId, tempOrderId.length() - 13);
			// 9452145845118
			break;
		case KUAIQIAN:
			tempOrderId = StringUtils.substring(tempOrderId, tempOrderId.length() - 13);
			// 1452145845125
			break;
		}
		BankPaylog payLog = new BankPaylog();
		payLog.setBankType(bankType);
		payLog.setSelfOrderId(orderidStr);
		payLog.setBankOrderId(tempOrderId);
		payLog.setAmount(amount);
		payLog.setCreateTm(new Date());
		payLog.setPayStatus(PayStatus.INIT);
		this.insert(payLog);
		return payLog;
	}

	private BankPaylog findByOrderidAndType(String id, BankType bankType) {
		return entityDao.findBySelfOrderIdAndBankType(id, bankType);
	}

	@Override
	public boolean isPaySuccess(Long id) {
		// 只是通过id查询
		return this.entityDao.countPaySuccess(id) > 0;
	}

	@Override
	public void updateStatusToSuccess(String bankOrderId, BankType banktype) {
		this.entityDao.updateStatusToSuccess(bankOrderId, banktype);
	}

	@Override
	public void updateStatusToFail(String bankOrderId, BankType banktype) {
		this.entityDao.updateStatusToFail(bankOrderId, banktype);
	}

	private BankPayLogService getProxy() {
		return (BankPayLogService) AopContext.currentProxy();
	}

	public boolean payCall(HttpServletRequest request, BankType bankType) {
		printMsg(request);
		boolean isOk = false;
		String orderno = null;
		switch (bankType) {
		case ALIPAY:
			orderno = request.getParameter("out_trade_no");
			isOk = AlipayInterfaceHelper.isPaySuccess(request);
			break;
		case CHINAPAY:
			orderno = request.getParameter("orderno");
			isOk = ChinaPayInterfaceHelper.isPaySuccess(request);
			break;
		case CHINABANK:
			orderno = request.getParameter("v_oid");
			isOk = ChinaBankPayHelper.isPaySuccess(request);
			break;
		case WECHAT:
			try {
				String xmlStr = IOUtils.toString(request.getInputStream());
				logger.info("微信支付回调请求xml:{}", xmlStr);
				NotifyResultParm result = XmlUtil.converyToJavaBean(xmlStr, NotifyResultParm.class);
				isOk = WxpayHelper.isPaySuccess(result);
				orderno = result.getOutTradeNo();
			} catch (Exception e) {
				e.printStackTrace();
			}
			break;
		case KUAIQIAN:
			orderno = request.getParameter("orderId");
			isOk = KuaiQianPayHelper.isPaySuccess(request);
			break;
		}
		if (StringUtils.isBlank(orderno)) {
			logger.info("订单号为空 ");
			return false;
		}
		getProxy().operatePayResult(bankType, isOk, orderno);
		logger.info("支付结果:{}", isOk);
		return isOk;
	}
	
	/**
	 * 支付结果的处理
	 */
	@Override
	@Transactional
	public void operatePayResult(BankType bankType, boolean payStatus, String outTradeNo) {
		BankPaylog bankPaylog = entityDao.findByBankOrderIdAndBankType(outTradeNo,bankType);
		if(bankPaylog != null){
			if (payStatus) {
				updateStatusToSuccess(outTradeNo, bankType);
				orderItemService.updateOrderState(Long.valueOf(bankPaylog.getSelfOrderId()), Status.PAID, AdmStatus.PAID, null);
			} else {
				updateStatusToFail(outTradeNo, bankType);
				orderItemService.updateOrderState(Long.valueOf(bankPaylog.getSelfOrderId()), Status.NOT_PAID, AdmStatus.NOT_PAID, null);
			}
		}
	}

	/**
	 * 打印参数。记录日志
	 * 
	 * @param request
	 * @author： 张文山
	 * @创建时间：2015-11-10 下午1:47:38
	 */
	private void printMsg(HttpServletRequest request) {
		StringBuilder sb = new StringBuilder();
		Map requestParams = request.getParameterMap();
		for (Iterator iter = requestParams.keySet().iterator(); iter.hasNext();) {
			String name = (String) iter.next();
			String[] values = (String[]) requestParams.get(name);
			String valueStr = "";
			for (int i = 0; i < values.length; i++) {
				valueStr = (i == values.length - 1) ? valueStr + values[i] : valueStr + values[i] + ",";
			}
			logger.info("{}------转码前:{}", name, valueStr);
		}
	}
}
