package com.rocoinfo.rocomall.service.impl;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Caching;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springside.modules.utils.Collections3;

import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.Address;
import com.rocoinfo.rocomall.entity.dict.DictProviceCity;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.repository.AddressDao;
import com.rocoinfo.rocomall.service.IAddressService;
import com.rocoinfo.rocomall.service.dict.IDictProviceCityService;
import com.rocoinfo.rocomall.service.order.IOrderService;
import com.rocoinfo.rocomall.utils.ValidateUtils;

/**
 * 用户收货地址DAO Created by robin on 14-6-11. author yangxc
 */
@SuppressWarnings("all")
@Component
@Transactional
public class AddressService extends CrudService<AddressDao, Address> implements IAddressService {

	@Autowired
	private IOrderService orderService;

	@Autowired
	private IDictProviceCityService dictService;

	public void insert(final Address address) {

		if (address != null) {
			if (!ValidateUtils.isMobile(address.getMobile())) {
				throw new ServiceException("收货人手机号不合法: " + address.getMobile());
			}

			super.insert(address);
			if (address.isDefaultAddr()) {
				setDefaultAddress(address.getId(), address.getUser().getId());
			}
		}
	}

	public void update(Address address) {
		if (address != null) {
			if (!ValidateUtils.isMobile(address.getMobile())) {
				throw new ServiceException("收货人手机号不合法: " + address.getMobile());
			}

			super.update(address);
			if (address.isDefaultAddr()) {
				setDefaultAddress(address.getId(), address.getUser().getId());
			}
		}
	}

	/**
	 * 只做逻辑删除 通过 用户id 和地址id 删除, 防止用户恶意传参 地址id删除
	 * 
	 * @param addrId
	 *            地址id
	 * @param userId
	 *            地址所属用户Id
	 */
	@Caching(evict = { @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ADDRESS_KEY_PREV + "#addrId"),
			@CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ADDRESS_ALL_KEY) })
	public void removeUserAddress(final Long addrId, final Long userId) {
		if (userId > 0 && addrId > 0) {
			Map<String, Object> paramMap = Maps.newHashMapWithExpectedSize(2);
			paramMap.put("id", addrId);
			paramMap.put("userId", userId);
			entityDao.removeUserAddress(paramMap);
		}
	}

	/**
	 * 通过用户Id获得他自己的收货地址列表
	 * 
	 * @param userId
	 *            用户id
	 * @return
	 */
	public List<Address> findUserAddressList(final Long userId, Boolean visiable) {
		List<Address> addrList = Collections.emptyList();
		if (userId > 0) {
			Map<String, Object> params = Maps.newHashMap();
			params.put("userId", userId);
			if (visiable != null) {
				params.put("visiable", visiable);
			}
			addrList = entityDao.findUserAddressList(params);
		}
		return addrList;
	}

	/**
	 * 收货地址查询 省市县
	 */
	public void buildProvCityCounty(final List<Address> addressLists) {

		for(Iterator<Address> it = addressLists.iterator();it.hasNext();){
			if (it.next()==null)
				it.remove();
		}

		if (!CollectionUtils.isEmpty(addressLists)) {
			Map<Long, List<DictProviceCity>> addrPrefixMap = dictService.buildParentProvCityNodeTree(false, addressLists.toArray(new Address[] {}));
			for (Address address : addressLists) {
				address.setProviceCityList(addrPrefixMap.get(address.getId()));
			}
		}
	}

	/**
	 * 设置用户默认收货地址
	 * 
	 * @param defAddrId
	 *            默认收货地址Id
	 * @param userId
	 *            用户Id
	 */
	@Caching(evict = { @CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ADDRESS_KEY_PREV + "#defAddrId"),
			@CacheEvict(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.ADDRESS_ALL_KEY) })
	public void setDefaultAddress(final Long defAddrId, final Long userId) {
		if (userId > 0 && defAddrId > 0) {
			Map<String, Object> paramMap = Maps.newHashMapWithExpectedSize(2);
			paramMap.put("id", defAddrId);
			paramMap.put("userId", userId);
			entityDao.setDefaultAddress(paramMap);
		}
	}

	@Override
	public List<Address> getAddressAndBuildProvCityCounty(Long userId) {
		List<Address> addressList = findUserAddressList(userId, null);
		if (Collections3.isNotEmpty(addressList)) {
			buildProvCityCounty(addressList);
		}
		return addressList;
	}
}
