package com.rocoinfo.rocomall.service.impl;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.ArrayUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springside.modules.utils.Collections3;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.entity.CartItem;
import com.rocoinfo.rocomall.repository.CartDao;
import com.rocoinfo.rocomall.service.ICartService;

/**
 * <dl>
 * <dd>描述: </dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2015/9/23 10:14</dd>
 * <dd>创建人：huyt</dd>
 * </dl>
 */
@Service
@Transactional
public class CartService extends CrudService<CartDao, CartItem> implements ICartService {

    public void saveCardItems(final CartItem... items) {
        if (!ArrayUtils.isEmpty(items)) {
            this.entityDao.saveCardItems(items);
        }
    }


    public List<CartItem> findCheckedCartItemListByUserId(final long userId) {
        List<CartItem> checkedItems = Lists.newArrayList();
        for (CartItem cartItem : this.findCartItemListByUserId(userId)) {
            if (cartItem.isChecked()) {
                checkedItems.add(cartItem);
            }
        }
        return checkedItems;
    }

    public List<CartItem> findCartItemListByUserId(final long userId) {
        List<CartItem> itemList = Collections.emptyList();
        if (userId > 0) {
            itemList = this.entityDao.getCartItemListByUserId(userId);
        }
        return itemList;
    }


    public List<CartItem> findCartItemListByUserAndSkuIdsIn(final long userId, List<Long> skuIds) {
        List<CartItem> itemList = Collections.emptyList();

        if (userId > 0 && !CollectionUtils.isEmpty(skuIds)) {
            itemList = Lists.newArrayList();

            List<CartItem> userCartItems = findCartItemListByUserId(userId);
            for (CartItem item : userCartItems) {
                if (skuIds.contains(item.getUnitProdId())) {
                    itemList.add(item);
                }
            }
        }
        return itemList;
    }


    public CartItem getCartItemByUnitProdId(final long userId, final long unitProdId) {
        CartItem item = null;
        for (CartItem itm : findCartItemListByUserId(userId)) {
            if (itm.getUnitProdId() == unitProdId) {
                item = itm;
                break;
            }
        }
        return item;
    }


    public void removeCartItemUnitProdId(final long userId, final Long... unitProdIds) {
        if (!ArrayUtils.isEmpty(unitProdIds) && userId > 0) {
            Map<String, Object> paramMap = Maps.newHashMapWithExpectedSize(1);
            paramMap.put("unitProdIds", unitProdIds);
            paramMap.put("userId", userId);
            this.entityDao.removeCartItem(paramMap);
        }
    }

    public void clearCartByUserId(final long userId) {
        if (userId > 0) {
            Map<String, Object> paramMap = Maps.newHashMapWithExpectedSize(1);
            paramMap.put("userId", userId);
            this.entityDao.removeCartItem(paramMap);
        }
    }

    /**
     * 修改cartItem的购买数量
     */
    public void modifyCardItemQuantity(final CartItem cartItem) {
        if (cartItem != null) {
            //需要的参数：unitProdId, userId, quantity
            this.entityDao.modifyCardItemQuantity(cartItem);
        }
    }


    /**
     * 更新用户选中购买的sku
     *
     * @param userId           用户Id
     * @param checkedSkuIdList 在购物车中被选中的skuId
     */
    public void updateCartItemCheckedIn(final long userId, List<Long> checkedSkuIdList) {
        if (userId > 0 && !CollectionUtils.isEmpty(checkedSkuIdList)) {
            String checkedSkuIds = Collections3.convertToString(checkedSkuIdList, ",");
            this.entityDao.updateCartItemCheckedIn(userId, checkedSkuIds);
        }
    }

    /**
     * 批量修改购物车 Item的数量
     *
     * @param cartItemQtMap<ItemId, quantity>
     */
    public void batchModifyCartItemQtMap(Map<Long, Integer> cartItemQtMap) {
        if (!CollectionUtils.isEmpty(cartItemQtMap)) {
            List<Long> itemIdList = Lists.newArrayListWithExpectedSize(cartItemQtMap.size());
            List<Integer> quantityList = Lists.newArrayListWithExpectedSize(cartItemQtMap.size());
            Iterator<Long> itemIdIter = cartItemQtMap.keySet().iterator();
            while (itemIdIter.hasNext()) {
                Long itemId = itemIdIter.next();
                itemIdList.add(itemId);

                quantityList.add(cartItemQtMap.get(itemId));
            }
            this.entityDao.batchModifyCartItemQts(itemIdList, quantityList);
        }
    }


    /**
     * 汇总购物车skus数量，为了替换CookieUtil.cookieCartItemMap()
     *
     * @return
     */
    public Map<Long, Integer> dbCartItemMap(Long loggedUserId) {
        Map<Long, Integer> itemQtMap = Maps.newLinkedHashMap();
        if (null == loggedUserId) {
            return itemQtMap;
        }
        List<CartItem> cartItemList = findCartItemListByUserId(loggedUserId);
        for (CartItem item : cartItemList) {
            long skuId = item.getUnitProdId();
            int quantity = item.getQuantity();
            itemQtMap.put(skuId, quantity);
        }
        return itemQtMap;
    }
}
