package com.rocoinfo.rocomall.service.impl;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springside.modules.utils.Collections3;

import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.entity.Address;
import com.rocoinfo.rocomall.entity.UserCatalogRelation;
import com.rocoinfo.rocomall.entity.account.User;
import com.rocoinfo.rocomall.entity.account.UserLevel;
import com.rocoinfo.rocomall.enumconst.SwitchStatus;
import com.rocoinfo.rocomall.enumconst.TTLUnit;
import com.rocoinfo.rocomall.repository.UserCatalogRelationDao;
import com.rocoinfo.rocomall.repository.account.UserDao;
import com.rocoinfo.rocomall.repository.cent.CentsDao;
import com.rocoinfo.rocomall.service.IUserService;
import com.rocoinfo.rocomall.utils.PasswordUtil;

/**
 * <dl>
 * <dd>描述: 会员Service</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2015年8月4日 上午10:14:48</dd>
 * <dd>创建人： 张敏</dd>
 * </dl>
 */
@Service
@Transactional
public class UserService extends CrudService<UserDao, User> implements IUserService {

	@Autowired
	private AddressService addressService;
	@Autowired
	private UserLevelService userLevelService;
	@Autowired
	private UserCatalogRelationDao userCatalogRelationDao;
	@Autowired
	private CentsDao centDao;

	/**
	 * 更新登录密码
	 */
	public void updateLoginPassword(final long userId, final String newPlainPwd) {
		if (userId > 0) {
			User user = new User(userId);
			user.setPlainPassword(newPlainPwd);
			PasswordUtil.entryptPassword(user);
			this.entityDao.update(user);
		}
	}

	/**
	 * 校验登录密码是否正确
	 * 
	 * @param username 登录名
	 * @param loginPassword 明文登录密码
	 */
	public boolean isLoginPasswordCorrect(final String username, final String loginPassword) {
		User user = getUserByUsername(username);
		if (user == null || user.getPassword() == null || user.getSalt() == null || loginPassword == null) {
			return false;
		}
		return PasswordUtil.hashPassword(loginPassword, user.getSalt()).equals(user.getPassword());
	}

	/**
	 * 判断用户名是否已经被注册
	 * 
	 * @param username 登录用户名
	 */
	public boolean isUsernameRegisted(String username) {
		return getUserByUsername(username) != null;
	}

	/**
	 * 成长值 和 账户积分余额不能用缓存里面的数据，因为这些数据可能实时变化
	 * 
	 * @return 返回成长值，如果userId不存在则返回-1
	 */
	public int getGrowthById(Long userId) {
		Integer growth = this.entityDao.getGrowthById(userId);
		return (growth == null) ? -1 : growth;
	}

	public void increaseGrowthById(Long userId, int increment) {
		if (userId == null || userId < 1 || increment == 0) {
			throw new IllegalArgumentException("userId必须大于0");
		}

		this.entityDao.increaseGrowthById(userId, increment);
	}

	/**
	 * 获取某个人 剩余积分余额
	 */
	public int getCentBalanceByUserId(Long userId) {
		return getCentBalanceByUserIdAndTTL(userId, TTLUnit.DAY, 0);
	}

	/**
	 * @param amount (Time To Live) 积分还剩多少天过期, ttlDays 0 表示不限制过期天数
	 */
	public int getCentBalanceByUserIdAndTTL(Long userId, TTLUnit ttlUnit, int amount) {
		Map<String, Object> params = Maps.newHashMap();
		params.put("userId", userId);
		if (amount > 0) {
			Date curDate = new Date();
			Date expireToDate = null;
			if (ttlUnit == TTLUnit.DAY) {
				expireToDate = DateUtils.addDays(curDate, amount);
			} else {
				expireToDate = DateUtils.addMonths(curDate, amount);
			}
			params.put("expireToDate", new java.sql.Date(expireToDate.getTime()));
		}
		return this.centDao.getCentBalanceByUserIdAndExpireTo(params);
	}

	public UserLevel getUserLevelById(Long userId) {
		int growth = getGrowthById(userId);
		if (growth >= 0) {
			return userLevelService.getUserLevelByGrowth(growth);
		}
		return null;
	}

	public User getUserByUsername(String username) {
		return entityDao.getByUsername(username);
	}

	public User getByMobile(String mobile) {
		return this.entityDao.getByMobile(mobile);
	}

	public Page<User> searchUser(Date registStartDate, Date registEndDate, String userNameOrName, String mobilePhone, SwitchStatus status, Pageable pageable) {

		Map<String, Object> parameters = Maps.newHashMap();

		if (registStartDate != null) {
			parameters.put("registStartDate", registStartDate);
		}

		if (registEndDate != null) {
			parameters.put("registEndDate", registEndDate);
		}

		if (status != null) {
			parameters.put("status", status);
		}

		if (StringUtils.isNotBlank(userNameOrName)) {
			parameters.put("userNameOrName", userNameOrName);
		}

		if (StringUtils.isNotBlank(mobilePhone)) {
			parameters.put("mobilePhone", mobilePhone);
		}

		parameters.put("sort", pageable.getSort());
		parameters.put("offset", pageable.getOffset());
		parameters.put("pageSize", pageable.getPageSize());

		long total = entityDao.searchTotal(parameters);
		List<User> users = Collections.emptyList();
		if (total > pageable.getOffset()) {
			users = entityDao.search(parameters);
		}
		return new PageImpl<User>(users, pageable, total);
	}

	/**
	 * 组装前台用户详细
	 */
	public void buildUserDetail(User user) {
		List<Address> addressList = addressService.findUserAddressList(user.getId(), null);
		if (Collections3.isNotEmpty(addressList)) {
			addressService.buildProvCityCounty(addressList);
			user.setAddressLists(addressList);
		}
	}

	public void bindMobileByUserId(long userId, final String mobile) {
		if (userId < 1 || StringUtils.isBlank(mobile)) {
			throw new IllegalArgumentException("userId必须大于零并且，mobile不能为空");
		}

		User user = new User();
		user.setId(userId);
		user.setMobilePhone(mobile);
		super.update(user);
	}

	public void switchUser(Long userId, SwitchStatus status) {
		User user = new User();
		user.setId(userId);
		user.setStatus(status);
		super.update(user);
	}

	@Override
	public List<Long> findCatalogIdsByUserId(Long loggedUserId) {
		return this.userCatalogRelationDao.findCatalogIdsByUserId(loggedUserId);
	}

	@Override
	public void personalSettingsSave(User user, Long[] catalogIds) {
		this.entityDao.update(user);
		saveUserCatalogRelations(user.getId(), catalogIds);
	}

	private void saveUserCatalogRelations(Long userId, Long[] catalogIds) {
		userCatalogRelationDao.deleteByUserId(userId);
		if (!ArrayUtils.isEmpty(catalogIds)) {
			for (Long catalogid : catalogIds) {
				userCatalogRelationDao.insert(new UserCatalogRelation(userId, catalogid));
			}
		}
	}
}
