package com.rocoinfo.rocomall.service.impl.dict;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.entity.Address;
import com.rocoinfo.rocomall.entity.dict.DictProviceCity;
import com.rocoinfo.rocomall.repository.dict.DictProviceCityDao;
import com.rocoinfo.rocomall.service.dict.IDictProviceCityService;

/**
 * 字典数据
 * 
 * @author zhangmin
 */
@SuppressWarnings("all")
@Component
@Transactional
public class DictProviceCityService extends CrudService<DictProviceCityDao, DictProviceCity> implements IDictProviceCityService {

	public Map<Long, DictProviceCity> loadAllDictProvCityMap() {
		Map<Long, DictProviceCity> map = Maps.newLinkedHashMap();
		for (final DictProviceCity provCity : this.findAll()) {
			map.put(provCity.getId(), provCity);
		}
		return map;
	}

	/**
	 * 构建省市节点列表
	 * 
	 * @param checkedCityIdList 勾选中的城市Id,如果无需勾选情况，则传入null
	 */
	public List<DictProviceCity> buildProvCityNodes(List<Long> checkedCityIdList) {

		Map<Long, DictProviceCity> allDictMap = loadAllDictProvCityMap();

		Map<Long, DictProviceCity> provMap = Maps.newLinkedHashMap();

		Iterator<Entry<Long, DictProviceCity>> iter = allDictMap.entrySet().iterator();
		while (iter.hasNext()) {
			Entry<Long, DictProviceCity> nodeEntry = iter.next();
			DictProviceCity provCityCounty = nodeEntry.getValue();

			if (provCityCounty.getPid() > 0) {
				DictProviceCity parent = allDictMap.get(provCityCounty.getPid());

				if (parent.getPid() == 0) {
					//provCityCounty是城市,parent是省
					if (!provMap.containsKey(parent.getId())) {
						provMap.put(parent.getId(), parent);
					}
					parent.addChild(provCityCounty);

					if (!CollectionUtils.isEmpty(checkedCityIdList) && checkedCityIdList.contains(provCityCounty.getId())) {
						provCityCounty.setSelected(true);
					}
				}
			}
		}
		return Lists.newArrayList(provMap.values());
	}

	/**
	 * @param isLoadNodeList 是否获取各层级的节点列表 构建所有祖先节点[包括自己]
	 */
	public Map<Long, List<DictProviceCity>> buildParentProvCityNodeTree(boolean isLoadNodeList, final Address... addressArray) {
		if (ArrayUtils.isNotEmpty(addressArray)) {
			Map<Long, DictProviceCity> allProvCity = loadAllDictProvCityMap();
			if (!CollectionUtils.isEmpty(allProvCity)) {
				Map<Long, List<DictProviceCity>> map = Maps.newHashMapWithExpectedSize(addressArray.length);
				for (final Address addr : addressArray) {
					if (addr != null && addr.getProvCity() != null) {
						map.put(addr.getId(), getParentProvCityNodeTree(allProvCity, addr.getProvCity(), isLoadNodeList));
					}
				}
				return map;
			}
		}
		return Collections.emptyMap();
	}

	/**
	 * 获得 和 provCity 同层级的节点列表
	 * 
	 * @param curLevelProvCity
	 */
	private void createSameLevelNodeList(Collection<DictProviceCity> allProvCity, DictProviceCity curLevelProvCity) {
		List<DictProviceCity> nodeList = Lists.newArrayList();
		for (DictProviceCity dict : allProvCity) {
			if (dict.getPid().equals(curLevelProvCity.getPid())) {

				//clone的目的是 防止 jackson  json序列化时出现死循环
				DictProviceCity clonedProvCity = new DictProviceCity();
				clonedProvCity.setId(dict.getId());
				clonedProvCity.setName(dict.getName());
				clonedProvCity.setDisplayOrder(dict.getDisplayOrder());
				clonedProvCity.setPid(dict.getPid());
				clonedProvCity.setDeliverFee(dict.getDeliverFee());
				clonedProvCity.setFreeLimitAmt(dict.getFreeLimitAmt());

				if (dict.getId().equals(curLevelProvCity.getId())) {
					clonedProvCity.setSelected(true);
				}
				nodeList.add(clonedProvCity);
			}
		}

		curLevelProvCity.setCurNodeList(nodeList);
	}

	private List<DictProviceCity> getParentProvCityNodeTree(Map<Long, DictProviceCity> allProvCityMap, DictProviceCity provCity, boolean isLoadNodeList) {
		final DictProviceCity dict = allProvCityMap.get(provCity.getId());
		try {
			BeanUtils.copyProperties(provCity, dict);
		} catch (Exception e) {
			e.printStackTrace();
		}

		Collection<DictProviceCity> allProvCityList = allProvCityMap.values();

		if (isLoadNodeList) {
			createSameLevelNodeList(allProvCityList, provCity);
		}

		List<DictProviceCity> parentNodes = Lists.newArrayList();

		DictProviceCity pNode = provCity;

		while (pNode.getPid() > 0) {
			pNode = allProvCityMap.get(pNode.getPid());
			parentNodes.add(pNode);

			if (isLoadNodeList) {
				createSameLevelNodeList(allProvCityList, pNode);
			}
		}

		if (!parentNodes.isEmpty()) {
			Collections.reverse(parentNodes);
		}

		parentNodes.add(provCity);

		return parentNodes;
	}

	public DictProviceCity getRootDictProviceCityById(final long id) {
		DictProviceCity provCity = super.getById(id);
		while (provCity != null && provCity.getPid() > 0) {
			provCity = super.getById(provCity.getPid());
		}
		return provCity;
	}

	public List<DictProviceCity> findDictProvCitysByPid(final Long pid) {
		if (pid == null) {
			return Collections.emptyList();
		}

		List<DictProviceCity> provCityList = Lists.newArrayList();
		for (DictProviceCity provCity : super.findAll()) {
			if (provCity.getPid().equals(pid)) {
				provCityList.add(provCity);
			}
		}

		return provCityList;
	}

	/**
	 * 批量修改省市 显示顺序
	 */
	@Deprecated
	public void batchModifyProvCitySeqs(final Integer[] ids, final Integer[] seqs) {
		if (ArrayUtils.isEmpty(ids) || ArrayUtils.isEmpty(seqs) || ids.length != seqs.length) {
			throw new IllegalArgumentException("批量修改 省市基础数据 顺序时，ids与seqs参数都不能为空，并且数组长度必须一致");
		}

		Map<String, Object> paramMap = Maps.newHashMap();
		paramMap.put("ids", ids);
		paramMap.put("seqs", seqs);

		entityDao.batchModifyProvCitySeqs(paramMap);
	}

	@Override
	public List<DictProviceCity> loadDictProvCityList(List<Long> pids) {
		if (pids != null && pids.size() > 0)
			return entityDao.findDictProvCitysByPids(pids);
		return null;
	}

}
