package com.rocoinfo.rocomall.service.impl.order;

import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Transactional;
import org.springside.modules.utils.Collections3;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.entity.Sku;
import com.rocoinfo.rocomall.entity.dict.DictWarehouse;
import com.rocoinfo.rocomall.entity.dict.express.DictExpress;
import com.rocoinfo.rocomall.entity.order.DistributionOrder;
import com.rocoinfo.rocomall.entity.order.MergeDistributeOrder;
import com.rocoinfo.rocomall.entity.order.Order;
import com.rocoinfo.rocomall.entity.order.OrderItem;
import com.rocoinfo.rocomall.entity.order.OrderItem.AdmStatus;
import com.rocoinfo.rocomall.entity.order.OrderItem.Status;
import com.rocoinfo.rocomall.entity.order.OrderOutStore;
import com.rocoinfo.rocomall.repository.order.DistributionOrderDao;
import com.rocoinfo.rocomall.service.order.IDistributionOrderService;
import com.rocoinfo.rocomall.utils.CodeGenerator;

/**
 * <dl>
 * <dd>描述:配送单Service实现</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>@创建时间：2015/11/12 17:50</dd>
 * <dd>@author：zhangmin</dd>
 * </dl>
 */
@SuppressWarnings("all")
@Service
public class DistributionOrderService extends CrudService<DistributionOrderDao, DistributionOrder> implements IDistributionOrderService {
	@Autowired
	private CodeGenerator codeGenerator;
	@Autowired
	private OrderService orderService;
	@Autowired
	private OrderOutStoreService outstoreService;

	/**
	 * 通过Id获取配送单,包括关联的订单Item，订单的配送地址信息一起查询出来
	 * 
	 * @param distriOrderIdList 配送单Id列表
	 */
	public List<DistributionOrder> findDistributeOrderWithItemAddrByIdIn(List<Long> distriOrderIdList) {
		if (CollectionUtils.isEmpty(distriOrderIdList))
			return Collections.emptyList();
		return this.entityDao.findDistributeOrderWithItemAddrByIdIn(distriOrderIdList);
	}

	/**
	 * 通过合并号 获得配送单Ids
	 * 
	 * @param mergeCode
	 */
	public List<Long> findDistributeOrderIdsByMergeCode(final String mergeCode) {
		if (StringUtils.isEmpty(mergeCode)) {
			return Collections.emptyList();
		}
		return this.entityDao.findDistributeOrderIdsByMergeCode(mergeCode);
	}

	/**
	 * 生成配送单
	 * 
	 * @param order 主订单：必须包含订单Item
	 */
	@Transactional(isolation = Isolation.READ_COMMITTED)
	public void generateDistributeOrder(Order order) {
		//只有实物订单才能生成配送单
		List<OrderItem> materialItems = Lists.newArrayList();
		for (OrderItem item : order.getItems()) {
			if (OrderItem.Group.MATERIAL.equals(item.getGroup())) {
				materialItems.add(item);
			}
		}

		if (CollectionUtils.isNotEmpty(materialItems)) {
			List<DistributionOrder> distriOrderList = Lists.newArrayListWithExpectedSize(materialItems.size());
			//配送单 合并的条件：收货人、收货人地址、仓库、快递公司都相同
			ListMultimap<String, DistributionOrder> orderGroupList = ArrayListMultimap.create();
			Date createTime = new Date();
			for (OrderItem itm : materialItems) {
				DistributionOrder distriOrder = new DistributionOrder();
				distriOrder.setCode(codeGenerator.generateDistributionOrderCode(itm.getOrderCode()));
				distriOrder.setExported(Boolean.FALSE);
				distriOrder.setOrderItem(itm);
				distriOrder.setMergeCode(StringUtils.EMPTY);
				//FIXME TODO modify 设置快递公司
				distriOrder.setExpress(itm.getExpress());
				distriOrder.setCreateTime(createTime);
				distriOrderList.add(distriOrder);

				if (itm.getExpress() != null && itm.getExpress().getId() > 0) {
					String wareHouseExpr = itm.getWareHouse().getId() + "-" + itm.getExpress().getId();
					orderGroupList.put(wareHouseExpr, distriOrder);
				}
			}

			Iterator<Collection<DistributionOrder>> iter = orderGroupList.asMap().values().iterator();
			while (iter.hasNext()) {
				Collection<DistributionOrder> group = iter.next();
				if (group.size() < 2) {
					continue;
				}

				//合并时,mergeCode更新为第一个的code
				String mergeCode = null;
				for (DistributionOrder grpItem : group) {
					if (mergeCode == null) {
						mergeCode = grpItem.getCode();
					}
					grpItem.setMergeCode(mergeCode);
				}
			}

			//保存配送单
			this.entityDao.batchSave(distriOrderList);
			//订单  更新为 已生成配送单
			Order orderToDistr = new Order();
			orderToDistr.setDistributed(Boolean.TRUE);
			orderToDistr.setId(order.getId());
			orderService.update(orderToDistr);
		}
	}

	/**
	 * 生成出库单
	 */
	@Transactional(isolation = Isolation.READ_COMMITTED)
	public void generateOutstoreOrder(List<DistributionOrder> distriOrderList, Long operatorId) {
		//出库单合并数量的条件为 配送单中的仓库、快递公司、商品编码相同
		if (CollectionUtils.isNotEmpty(distriOrderList)) {
			//key格式：仓库Id + 快递Id +skuCode
			String keyTemplate = "%d-%d-%s";
			ListMultimap<String, DistributionOrder> multiMap = ArrayListMultimap.create();
			for (DistributionOrder distriOrder : distriOrderList) {
				DictWarehouse wareHouse = distriOrder.getOrderItem().getWareHouse();
				DictExpress express = distriOrder.getExpress();
				String skuCode = distriOrder.getOrderItem().getSku().getCode();
				String key = String.format(keyTemplate, wareHouse.getId(), express.getId(), skuCode);
				multiMap.put(key, distriOrder);
			}

			//配送单 需要更新为已配送导出,并更新相应的出库单号
			List<Long> distriOrderIds = Lists.newArrayList();
			List<Long> outStoreIds = Lists.newArrayList();

			Iterator<String> iter = multiMap.keySet().iterator();
			while (iter.hasNext()) {
				List<DistributionOrder> ordList = multiMap.get(iter.next());
				OrderOutStore outstoreOrder = null;
				int quantityTotal = 0;
				Date createTime = new Date();
				for (DistributionOrder ord : ordList) {
					if (outstoreOrder == null) {
						outstoreOrder = new OrderOutStore();
						outstoreOrder.setCode(codeGenerator.generateOutstoreOrderCode());
						outstoreOrder.setCreateTime(createTime);
						outstoreOrder.setExpress(ord.getExpress());
						outstoreOrder.setState(Boolean.FALSE);
						Sku sku = ord.getOrderItem().getSku();
						if (StringUtils.isEmpty(sku.getName())) {
							sku.setName(ord.getOrderItem().getProductName());
						}
						outstoreOrder.setSku(sku);
						outstoreOrder.setWarehouse(ord.getOrderItem().getWareHouse());
					}

					distriOrderIds.add(ord.getId());
					quantityTotal += ord.getOrderItem().getQuantity();
				}

				outstoreOrder.setQuantity(quantityTotal);
				outstoreService.insert(outstoreOrder);
				for (int i = 0; i < ordList.size(); i++) {
					outStoreIds.add(outstoreOrder.getId());
				}
			}

			//更新配送单
			this.entityDao.updateExportedAndOutstoreIds(distriOrderIds, outStoreIds, operatorId);
			//通过出库单id,更新子订单状态
			for(Long id : outStoreIds){
				outstoreService.upDateOrderItemStatus(OrderItem.Status.PAID, OrderItem.AdmStatus.DISTRIBUTION_WAIT_OUT, id);	
			}
		}
	}

	/**
	 * 合并/拆分 配送单
	 * 
	 * @param distriOrderIdList
	 * @param mergeCode mergeCode为空表示拆分,不空表示合并
	 */
	public void mergeOrSplit(List<Long> distriOrderIdList, String mergeCode) {
		if (CollectionUtils.isNotEmpty(distriOrderIdList)) {
			this.entityDao.mergeOrSplit(distriOrderIdList, mergeCode);
		}
		if(mergeCode != null && !mergeCode.trim().equals("")){
			this.entityDao.updateOrderItemByIds(distriOrderIdList, Status.PAID, AdmStatus.DISTRIBUTION_MERGED);
		}
	}

	/**
	 * 导入 物流单号
	 */
	@Transactional(isolation = Isolation.READ_COMMITTED)
	public void importLogistics(List<DistributionOrder> distriOrderList) {
		if (CollectionUtils.isNotEmpty(distriOrderList)) {
			for (DistributionOrder distrOrder : distriOrderList) {
				this.entityDao.update(distrOrder);
			}
			List<String> distriOrderCodeList = Collections3.extractToList(distriOrderList, "code");
			this.entityDao.updateOrderItemToDelivered(distriOrderCodeList,Status.RECEIVING,AdmStatus.DELIVERED,false);
		}
	}
	
	/**
	 * 更新子订单状态
	 */
	@Transactional(isolation = Isolation.READ_COMMITTED)
	public void updateOrderItemStatus(List<DistributionOrder> distriOrderList,OrderItem.Status status, OrderItem.AdmStatus admStatus) {
		if (CollectionUtils.isNotEmpty(distriOrderList)) {
			List<String> distriOrderCodeList = Collections3.extractToList(distriOrderList, "code");
			this.entityDao.updateOrderItemToDelivered(distriOrderCodeList,status,admStatus,true);
		}
	}

	/**
	 * 搜索符合 合并条件的 配送单
	 */
	public Page<MergeDistributeOrder> searchMergeDistributeOrder(Map<String, Object> params, Pageable pageable) {
		params.put(Constants.PAGE_OFFSET, pageable.getPageNumber());
		params.put(Constants.PAGE_SIZE, pageable.getPageSize());
		params.put(Constants.PAGE_SORT, pageable.getSort());

		List<MergeDistributeOrder> pageData = Collections.emptyList();
		Long count = this.entityDao.searchTotalMergeDistriOrder(params);
		if (count > 0) {
			pageData = entityDao.searchMergeDistriOrder(params);
		}
		return new PageImpl<MergeDistributeOrder>(pageData, pageable, count);
	}

}
