package com.rocoinfo.rocomall.service.impl.product;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.google.common.collect.Lists;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.Catalog;
import com.rocoinfo.rocomall.entity.Catalog.Status;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.repository.CatalogDao;
import com.rocoinfo.rocomall.service.impl.UploadService;
import com.rocoinfo.rocomall.service.product.ICatalogService;

@Service
public class CatalogService extends CrudService<CatalogDao, Catalog> implements ICatalogService {

	@Autowired
	private UploadService uploadService;

	@Override
	@Cacheable(value = CacheKeys.DEFAULT_NAME, key = CacheKeys.CATALOG_ALL_KEY)
	public List<Catalog> findAll() {
		return super.findAll();
	}

	@Override
	protected void evictCacheEntitysHook() {
		String allCatalogCacheKey = StringUtils.substringBetween(CacheKeys.CATALOG_ALL_KEY, "'");
		Cache cache = cacheManager.getCache(CacheKeys.DEFAULT_NAME);
		cache.evict(allCatalogCacheKey);
	}

	public void save(Catalog catalog) {
		if (catalog.getId() == null) {
			catalog.setId(0l);// 默认为root节点
		}
		if (isExistCatalogName(catalog)) {
			throw new ServiceException("分类名称已经存在!");
		}
		if (StringUtils.isNotBlank(catalog.getImgUrl())) {
			catalog.setImgUrl(uploadService.submitPath(catalog.getImgUrl()));
		}
		super.insert(catalog);
	}

	@Override
	public void update(Catalog catalog) {
		if (isExistCatalogName(catalog)) {
			throw new ServiceException("分类名称已经存在!");
		}
		Catalog oldCatalog = this.getById(catalog.getId());
		if (StringUtils.isNotBlank(catalog.getImgUrl())) {
			catalog.setImgUrl(uploadService.updateFile(catalog.getImgUrl(), oldCatalog.getImgUrl()));
		}

		super.update(catalog);
	}

	/**
	 * 查询子分类列表
	 * 
	 * @param parentId 父分类Id, 0表示查询顶级分类
	 * @param states 分类显示状态，如果是多个状态，则返回多个状态值分类的并集, states是null或空数组,则返回所有状态的分类
	 */
	public List<Catalog> findSubCatalogs(final long parentId, Catalog.Status... states) {
		List<Catalog> subCatalogs = Collections.emptyList();

		final List<Catalog> allCatalogs = getCurrentProxy().findAll();

		if (!CollectionUtils.isEmpty(allCatalogs)) {
			subCatalogs = Lists.newArrayList();
			for (final Catalog cate : allCatalogs) {
				if (ArrayUtils.isEmpty(states) || ArrayUtils.contains(states, cate.getStatus())) {
					if (parentId > 0) {
						if (cate.getParent() != null && cate.getParent().getId() == parentId) {
							subCatalogs.add(cate);
						}
					} else {
						if (cate.getParent() == null || cate.getParent().getId() == null || cate.getParent().getId() < 1) {
							subCatalogs.add(cate);
						}
					}
				}

			}
		}
		return subCatalogs;
	}

	/**
	 * 构建分类树
	 * 
	 * @isRequireTreeLine 是否需要树分割线
	 */
	public List<Catalog> buildCatalogTree(boolean isRequireTreeLine) {
		final List<Catalog> allCatalogs = getCurrentProxy().findAll();
		if (!CollectionUtils.isEmpty(allCatalogs)) {

			final List<Catalog> prodTypeTree = Lists.newArrayList();

			for (final Catalog prodType : findTopCatalogs(allCatalogs)) {
				recurse(prodType, prodTypeTree, allCatalogs, 0, isRequireTreeLine);
			}
			return prodTypeTree;
		}

		return allCatalogs;
	}

	private void recurse(final Catalog curCatalog, final List<Catalog> tree, final List<Catalog> allCatalog, int depth, boolean isRequireTreeLine) {
		if (curCatalog != null) {
			if (isRequireTreeLine) {
				String name = curCatalog.getName();
				if (depth == 0) {
					name = "╋" + name;
				} else {
					String blank = "";
					for (int i = 0; i < depth; i++) {
						blank += "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
					}
					name = blank + "├" + name;
				}
				curCatalog.setName(name);
			}

			tree.add(curCatalog);

			List<Catalog> childCateList = findSubCatelogs(curCatalog.getId(), allCatalog);
			Iterator<Catalog> iter = childCateList.iterator();
			while (iter.hasNext()) {
				recurse(iter.next(), tree, allCatalog, depth + 1, isRequireTreeLine);
			}
		}
	}

	private List<Catalog> findSubCatelogs(final long parentId, final List<Catalog> allCatalogs) {
		final List<Catalog> subCateList = Lists.newArrayList();
		for (final Catalog catalog : allCatalogs) {
			if (catalog != null && catalog.getParent() != null && catalog.getParent().getId() == parentId) {
				subCateList.add(catalog);
			}
		}
		return subCateList;
	}

	/**
	 * 所有的顶级分类
	 * 
	 * @param allCatalogs 系统中所有的分类
	 */
	private List<Catalog> findTopCatalogs(List<Catalog> allCatalogs) {
		if (CollectionUtils.isEmpty(allCatalogs)) {
			return allCatalogs;
		}

		List<Catalog> topCatalogs = Lists.newArrayList();

		for (final Catalog cate : allCatalogs) {
			if (cate != null && (cate.getParent() == null || cate.getParent().getId() < 1)) {
				topCatalogs.add(cate);
			}
		}
		return topCatalogs;
	}

	public List<Catalog> findByCatalogIds(List<Long> catalogIds) {
		List<Catalog> productCatalogs = new ArrayList<Catalog>();
		for (Long catalogId : catalogIds) {
			productCatalogs.add(getById(catalogId));
		}
		return productCatalogs;
	}

	/**
	 * @param id Catalog ID
	 * @return 返回包含指定节点ID和所有后代节点ID的List
	 */
	public List<Long> findSubTreeIds(Long id) {
		List<Catalog> catalogs = this.getCurrentProxy().findAll();

		Map<Long, Catalog> idToCatalog = new HashMap<Long, Catalog>(catalogs.size());
		for (Catalog catalog : catalogs) {
			idToCatalog.put(catalog.getId(), catalog);
		}

		List<Long> subTreeIds = new ArrayList<Long>();
		subTreeIds.add(id);

		// 遍历所有条目，找出后代ID
		for (Catalog catalog : catalogs) {
			if (id.equals(catalog.getId())) {
				continue;
			}

			Long parentId = catalog.getParentId();
			while (parentId != null) {
				if (parentId.equals(id)) {
					subTreeIds.add(catalog.getId());
					break;
				} else {
					try {
						parentId = idToCatalog.get(parentId).getParentId();
					} catch (Exception e) {
						break;
					}

				}
			}
		}

		return subTreeIds;
	}

	/**
	 * 返回所有Catalog的树形结构.
	 * 
	 * @param isAll 是否显示全部，如果为false 则屏蔽掉都不可见的分类
	 */
	public List<Catalog> tree(boolean isAll) {
		List<Catalog> catalogs = getCurrentProxy().findAll();

		Map<Long, Catalog> catalogMap = new HashMap<Long, Catalog>(catalogs.size());
		for (Catalog catalog : catalogs) {
			if (!isAll && catalog.getStatus() == Status.INVISIBLE) {
				continue;
			}
			catalogMap.put(catalog.getId(), catalog);
		}

		List<Catalog> root = new ArrayList<Catalog>();
		for (Catalog catalog : catalogs) {
			if (!isAll && catalog.getStatus() == Status.INVISIBLE) {
				continue;
			}

			if (catalog.getParentId() == null) {
				root.add(catalog);
			} else {
				if (catalogMap.get(catalog.getParentId()) == null) {
					continue;
				}
				List<Catalog> children = catalogMap.get(catalog.getParentId()).getChildren();
				if (children == null) {
					children = new ArrayList<Catalog>();
					catalogMap.get(catalog.getParentId()).setChildren(children);
				}
				children.add(catalog);
			}
		}
		return root;
	}

	/**
	 * 根据name条件查询分类
	 * 
	 * @param parame
	 * @return
	 */
	public List<Long> findCatalogIdsbyName(String name) {
		List<Long> list = new ArrayList<Long>();
		if (name == null || name.equals(""))
			return null;
		Map<String, Object> params = new HashMap<String, Object>();
		params.put("name", name);
		params.put("offset", 0);
		params.put("pageSize", Integer.MAX_VALUE);
		params.put("parentId", 0);
		List<Catalog> search = entityDao.search(params);
		Set<Long> set = new HashSet();
		if (search != null && search.size() > 0) {
			for (Catalog c : search) {
				List<Long> findSubTreeIds = this.findSubTreeIds(c.getId());
				set.addAll(findSubTreeIds);
			}
		} else {
			return null;
		}
		List<Long> result = new ArrayList<Long>();
		result.addAll(set);
		return result;
	}

	/**
	 * 判断同级目录是否存在同名的兄弟姐妹
	 */
	private boolean isExistCatalogName(Catalog catalog) {
		if (catalog == null || StringUtils.isBlank(catalog.getName())) {
			throw new ServiceException("分类名称不能为空!");
		}

		for (Catalog cate : this.getCurrentProxy().findAll()) {
			if (catalog.getId() != null && catalog.getId() == catalog.getId() && StringUtils.equals(cate.getName(), catalog.getName())) {
				// 做更新操作。id,name都没有修改,说明不存在重复的分类
				return false;
			}
			if (cate.getName().equals(catalog.getName())) {
				return true;
			}
		}
		return false;
	}

	private CatalogService getCurrentProxy() {
		return (CatalogService) AopContext.currentProxy();
	}

}
