package com.rocoinfo.rocomall.service.impl.product;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springside.modules.utils.Collections3;

import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.service.CrudService;
import com.rocoinfo.rocomall.common.service.ServiceException;
import com.rocoinfo.rocomall.entity.Catalog;
import com.rocoinfo.rocomall.entity.ProdTemp;
import com.rocoinfo.rocomall.entity.Product;
import com.rocoinfo.rocomall.entity.ProductImage;
import com.rocoinfo.rocomall.entity.ProductImage.ImageType;
import com.rocoinfo.rocomall.entity.Sku;
import com.rocoinfo.rocomall.entity.account.AdminLog;
import com.rocoinfo.rocomall.entity.account.User;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.redis.JedisTemplate;
import com.rocoinfo.rocomall.repository.ProdTempDao;
import com.rocoinfo.rocomall.repository.ProductDao;
import com.rocoinfo.rocomall.repository.SkuDao;
import com.rocoinfo.rocomall.service.IAdminLogService;
import com.rocoinfo.rocomall.service.product.IBrandService;
import com.rocoinfo.rocomall.service.product.IDescriptionService;
import com.rocoinfo.rocomall.service.product.IProductCatalogService;
import com.rocoinfo.rocomall.service.product.IProductImageService;
import com.rocoinfo.rocomall.service.product.IProductService;
import com.rocoinfo.rocomall.service.product.ISkuMetaService;
import com.rocoinfo.rocomall.service.product.ISkuService;
import com.rocoinfo.rocomall.service.product.ISupplierService;

/**
 * <dl>
 * <dd>描述:</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>@创建时间：2015-8-3 下午2:42:13</dd>
 * <dd>@author： 张文山</dd>
 * </dl>
 */
@Service
public class ProductService extends CrudService<ProductDao, Product> implements IProductService {

    @Autowired
    private JedisTemplate jedisTemplate;
    @Autowired
    private IDescriptionService descriptionService;
    @Autowired
    private ISkuMetaService skuMetaService;
    @Autowired
    private ISkuService skuService;
    @Autowired
    private ISupplierService supplierService;
    @Autowired
    private IBrandService brandService;
    @Autowired
    private IProductImageService productImageService;
    @Autowired
    private SkuDao skuDao;

    @Autowired
    private ProdTempDao prodTempDao;

    @Autowired
    private IProductCatalogService productCatalogService;
    @Autowired
    private IAdminLogService adminLogService;

    public void buildDetail(Product product) {
        product.setBrand(brandService.getById(product.getBrand().getId()));
        List<Catalog> catalogs = productCatalogService.findCatalogByProductId(product.getId());
        if (CollectionUtils.isNotEmpty(catalogs)) {
            product.setCatalogs(catalogs);
        }
        product.setSupplier(supplierService.getById(product.getSupplier().getId()));
        product.setProductImages(productImageService.getProductPrimaryImages(product.getId()));
        product.setSkuMeta(skuMetaService.getByProductIdAndDictMeta(product.getId()));

        List<Sku> skus = skuDao.getByProductId(product.getId());
        product.setSkus(skus);
        for (Sku sku : skus) {
            sku.setProductImages(productImageService.getBySkuId(sku.getId()));
        }
        product.setDescription(descriptionService.findByProductId(product.getId()));

    }

    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void create(Product product, Long userId) {
        if (!product.getListTime().before(product.getDelistTime())) {
            throw new ServiceException("下架时间应该大于上架时间");
        }
        if (Collections3.isEmpty(product.getSkus())) {
            throw new ServiceException("子商品不能为空");
        }
        if (product.getDescription() == null) {
            throw new ServiceException("商品描述不能为空");
        }
        if (product.getVisual() && product.getPutType() == null) {
            throw new ServiceException("虚拟商品必须选择投放方式");
        } else if (!product.getVisual()) {
            product.setPutType(null);
        }
        updateProductLogicValue(product, true);
        super.insert(product);

        if (product.getCatalogs() != null) {
            for (Catalog catalog : product.getCatalogs()) {
                productCatalogService.insert(product.getId(), catalog.getId());
            }
        }
        List<ProductImage> mainImages = product.getProductImages();
        if (mainImages != null) {
            int index = 0;
            for (ProductImage image : mainImages) {
                if (StringUtils.isBlank(image.getPath())) {
                    continue;
                }
                image.setDisplayOrder(index);
                index++;
                image.setProduct(product);
                if (image.getImageType() == null) {
                    image.setImageType(ImageType.ATTACH);
                }
                productImageService.insert(image);
            }
        }

        product.getDescription().setProductId(product.getId());
        descriptionService.insert(product.getDescription());

        if (product.getSkuMeta() != null) {
            product.getSkuMeta().setProduct(product);
            skuMetaService.insert(product.getSkuMeta());
        }

        int i = 0;
        for (Sku sku : product.getSkus()) {
            sku.setCode(product.getId() + StringUtils.leftPad(String.valueOf(i + 1), 2, '0'));
            i++;
            skuService.insert(sku);
        }
        insertAdminLog(product.getId(), AdminLog.Action.NEW, userId);
    }

    /**
     * 更新商品 (后台专用)。
     * <p/>
     * 注意，要同时设置上架时间和下架时间，否则本方法不保证下架时间大于上架时间
     *
     * @throws IllegalArgumentException 如果品ID为空; 如果下架时间不大于上架时间
     */
    @Transactional(isolation = Isolation.SERIALIZABLE, propagation = Propagation.REQUIRED)
    public void update(Product product, Long userId) {
        if (product.getId() == null) {
            throw new ServiceException("商品id不能为空");
        }
        if (product.getVisual() && product.getPutType() == null) {
            throw new ServiceException("虚拟商品必须选择投放方式");
        } else if (!product.getVisual()) {
            product.setPutType(null);
        }
        Product oldProduct = entityDao.getById(product.getId());
        if (oldProduct.getStatus() != Product.Status.DRAFT && !canChangeStatus(oldProduct.getStatus(), Product.Status.DRAFT)) {
            throw new ServiceException("当前状态 " + oldProduct.getStatus().getLabel() + " 不可编辑");
        }
        // 编辑商品保存后变为草稿状态
        product.setStatus(Product.Status.DRAFT);

        if (!product.getListTime().before(product.getDelistTime())) {
            throw new ServiceException("下架时间应该大于上架时间");
        }
        updateProductLogicValue(product, false);
        super.update(product);
        // 更新类别
        if (CollectionUtils.isNotEmpty(product.getCatalogs())) {
            productCatalogService.deleteByProductId(product.getId());
            for (Catalog catalog : product.getCatalogs()) {
                productCatalogService.insert(product.getId(), catalog.getId());
            }
        }

        // 更新图片
        if (CollectionUtils.isNotEmpty(product.getProductImages())) {
            List<ProductImage> oldProductImages = productImageService.getProductPrimaryImages(product.getId());
            productImageService.batchUpdate(oldProductImages, product.getProductImages(), product);
        }

        if (product.getDescription() != null) {
            product.getDescription().setProductId(product.getId());
            descriptionService.update(product.getDescription());
        }

        if (product.getSkuMeta() != null) {
            skuMetaService.update(product.getSkuMeta());
        }

        if (product.getSkus() != null) {
            for (Sku sku : product.getSkus()) {
                sku.setProduct(product);
                skuService.update(sku, userId);
            }
        }
        insertAdminLog(product.getId(), AdminLog.Action.EDIT, userId);
    }

    /**
     * 更新主商品冗余字段
     *
     * @param product
     */
    private void updateProductLogicValue(Product product, boolean isInsert) {
        Double cashAmt = 0d; // 现金价值
        Double cent = 0d; // 积分价值
        Double mergeCashAmt = 0d; // 合并现金价值
        Double mergeCent = 0d; // 合并积分价值
        for (int i = 0; i < product.getSkus().size(); i++) {
            Sku sku = product.getSkus().get(i);
            sku.setProduct(product);
            if (sku.getCashAmt() == null) {
                sku.setCashAmt(0d);
            }
            if (sku.getCent() == null) {
                sku.setCent(0);
            }
            if (sku.getMergeCashAmt() == null) {
                sku.setMergeCashAmt(0d);
            }
            if (sku.getMergeCent() == null) {
                sku.setMergeCent(0d);
            }
            if (sku.getCashAmt() > 0) {
                // 将sku中最小值放在product中
                cashAmt = Math.min(cashAmt, sku.getCashAmt());
            }
            if (sku.getCent() > 0) {
                cent = Math.min(cent, sku.getCent());
            }
            if (sku.getMergeCashAmt() > 0) {
                mergeCashAmt = Math.min(mergeCashAmt, sku.getMergeCashAmt());
            }
            if (sku.getMergeCent() > 0) {
                mergeCent = Math.min(mergeCent, sku.getMergeCent());
            }
            if (isInsert) {
                sku.setDisplayInlist(true);
            }
        }
        if (isInsert) {
            product.setDisplayInlist(true);
        }
        product.setCashAmt(cashAmt);
        product.setCent(cent);
        product.setMergeCashAmt(mergeCashAmt);
        product.setMergeCent(mergeCent);
        // 是否在列表页显示
    }

    private void insertAdminLog(Long id, AdminLog.Action action, Long userId) {
        adminLogService.insert(new AdminLog(userId, AdminLog.ObjectType.PRODUCT, id, action));
    }

    public List<Long> scheduleList() {
        try {
            Date date = new Date();
            List<Long> productIds = entityDao.findScheduleReadyListProductIds(date);
            if (CollectionUtils.isNotEmpty(productIds)) {
                entityDao.updateProductStatus(Product.Status.LIST, productIds);
                removeCached(productIds);
            }
            return productIds;
        } catch (Exception e) {
            logger.error("产品上架失败.error:{}", e);
            e.printStackTrace();
            return Collections.emptyList();
        }
    }

    public List<Long> scheduleDelist() {
        try {
            Date date = new Date();
            List<Long> productIds = entityDao.findScheduleDeListProductIds(date);
            if (CollectionUtils.isNotEmpty(productIds)) {
                entityDao.updateProductStatus(Product.Status.DELIST, productIds);
                removeCached(productIds);
            }
            return productIds;
        } catch (Exception e) {
            logger.error("产品下架失败.error:{}", e);
            e.printStackTrace();
            return Collections.emptyList();
        }
    }

    private void removeCached(List<Long> ids) {
        for (Long id : ids) {
            // 清空缓存
            jedisTemplate.del(CacheKeys.PRODUCT_KEY_PREFIX + id);
        }
    }

    public boolean canChangeStatus(Product.Status from, Product.Status to) {

        if (from == Product.Status.DRAFT) {
            return to == Product.Status.DISCARD || to == Product.Status.AUDIT;
        }
        if (from == Product.Status.DISCARD) {
            return false;
        }
        if (from == Product.Status.AUDIT) {
            return to == Product.Status.REFUSE || to == Product.Status.READY_LIST || to == Product.Status.DRAFT;
        }
        if (from == Product.Status.REFUSE) {
            return to == Product.Status.DISCARD || to == Product.Status.DRAFT;
        }
        if (from == Product.Status.LIST) {
            return to == Product.Status.DELIST;
        }
        if (from == Product.Status.READY_LIST) {
            return to == Product.Status.DRAFT || to == Product.Status.LIST;
        }
        if (from == Product.Status.DELIST) {
            return to == Product.Status.LIST || to == Product.Status.DRAFT;
        }
        return false;
    }

    /**
     * 修改商品状态.
     *
     * @throws com.rocoinfo.exception.ServiceException 如果不可从当前状态改为目标状态
     */
    private void changeStatus(long id, Product.Status newStatus) {
        Product product = getProxy().getById(id);

        if (!canChangeStatus(product.getStatus(), newStatus)) {
            throw new ServiceException("商品状态不能从 " + product.getStatus().getLabel() + " 改为 " + newStatus.getLabel());
        }
        Product upProduct = new Product();
        upProduct.setId(id);
        upProduct.setStatus(newStatus);
        super.update(upProduct);
    }

    /**
     * 弃用
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void discard(long id, Long userId) {
        insertAdminLog(id, AdminLog.Action.DISCARD, userId);
        changeStatus(id, Product.Status.DISCARD);
    }

    /**
     * 提交审核
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void submitAudit(Long id, Long userId) {
        insertAdminLog(id, AdminLog.Action.SUBMIT_AUDIT, userId);
        changeStatus(id, Product.Status.AUDIT);
    }

    /**
     * 审核拒绝
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void auditRefuse(Long id, Long userId) {
        insertAdminLog(id, AdminLog.Action.AUDIT_REFUSE, userId);
        changeStatus(id, Product.Status.REFUSE);
    }

    /**
     * 审核通过
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void auditOk(Long id, Long userId) {
        insertAdminLog(id, AdminLog.Action.AUDIT_OK, userId);
        changeStatus(id, Product.Status.READY_LIST);
    }

    /**
     * 手动上架. 如果定时下架时间小于当前时间，则清空定时下架时间
     *
     * @throws com.rocoinfo.exception.ServiceException 如果当前商品状态不可上架
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void list(Long id, Long userId) throws Exception {
        Product product = getProxy().getById(id);
        if (!canChangeStatus(product.getStatus(), Product.Status.LIST)) {
            throw new ServiceException("商品状态不能从 " + product.getStatus().getLabel() + " 改为 " + Product.Status.LIST.getLabel());
        }

        // 清空定时下架时间
        Date now = new Date();
        if (product.getDelistTime() != null && product.getDelistTime().getTime() < now.getTime()) {
            throw new ServiceException("商品下架时间早于当前时间，请重新编辑商品下架时间！");
        }

        // 更新商品状态
        Product productUpdate = new Product(id);
        productUpdate.setStatus(Product.Status.LIST);
        super.update(productUpdate);
        insertAdminLog(id, AdminLog.Action.LIST, userId);
    }

    /**
     * 手动下架.
     *
     * @throws com.rocoinfo.exception.ServiceException 如果当前商品状态不可下架
     */
    @Transactional(isolation = Isolation.READ_COMMITTED, propagation = Propagation.REQUIRED)
    public void delist(Long id, Long userId) {
        Product product = getProxy().getById(id);
        if (!canChangeStatus(product.getStatus(), Product.Status.DELIST)) {
            throw new ServiceException("商品状态不能从 " + product.getStatus().getLabel() + " 改为 " + Product.Status.LIST.getLabel());
        }

        // 更新商品状态
        Product productUpdate = new Product(id);
        productUpdate.setStatus(Product.Status.DELIST);
        super.update(productUpdate);
        insertAdminLog(id, AdminLog.Action.DELIST, userId);
    }

    public long countListedProductBySupplierId(long supplierId) {
        return entityDao.countListedProductBySupplierId(supplierId);
    }

    @Transactional(isolation = Isolation.READ_COMMITTED, readOnly = true)
    public Page<Product> search(Map<String, Object> params, Pageable pageable) {
        Long total = entityDao.searchTotal(params);
        // List<Product> content;
        if (total > pageable.getOffset()) {
            params.put(Constants.PAGE_OFFSET, pageable.getOffset());
            params.put(Constants.PAGE_SIZE, pageable.getPageSize());
            params.put(Constants.PAGE_SORT, pageable.getSort());

            // List<Long> ids = productDao.search(params);

			/*
             * content = new ArrayList<Product>(ids.size()); for (Long id : ids)
			 * { Product product = getProxy().get(id);
			 * product.setSupplier(supplierService
			 * .get(product.getSupplier().getId()));
			 * product.setBrand(brandService.get(product.getBrand().getId()));
			 * product
			 * .setCatalogs(productCatalogService.findCatalogByProductId(product
			 * .getId()));
			 * product.setSkus(skuService.getByProductId(product.getId()));
			 * product
			 * .setProductImages(productImageService.getProductPrimaryImages
			 * (product.getId())); }
			 */
            /*
             * } else { content = Collections.emptyList(); }
			 */
        }
        // return new PageImpl<Product>(content, pageable, total);
        return null;
    }

    private ProductService getProxy() {
        return (ProductService) AopContext.currentProxy();
    }

    @Override
    public Long createTemp(String jsonStr, Long userId) {
        if (StringUtils.isBlank(jsonStr)) {
            throw new ServiceException("商品数据不能为空");
        }
        if (userId == null) {
            throw new ServiceException("用户没有登录");
        }
        ProdTemp temp = new ProdTemp();
        User user = new User();
        user.setId(userId);
        temp.setUser(user);
        temp.setJsonString(jsonStr);
        prodTempDao.insert(temp);
        return temp.getId();
    }

    @Override
    public void updateTemp(String jsonStr, Long id) {
        if (StringUtils.isBlank(jsonStr)) {
            throw new ServiceException("商品数据不能为空");
        }
        ProdTemp temp = new ProdTemp();
        temp.setId(id);
        temp.setJsonString(jsonStr);
        prodTempDao.update(temp);
    }

    @Override
    public ProdTemp getLastProdTempByUserId(Long userId) {
        return prodTempDao.getByUserId(userId);
    }

    @Override
    public void updateIsInListStatus(Long productId) {
        entityDao.updateIsInListStatus(productId);
    }
}
