/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.rocoinfo.rocomall.common.service;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.Cache.ValueWrapper;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.redis.cache.RedisCacheManager;

import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.common.persistence.CrudDao;
import com.rocoinfo.rocomall.entity.IdEntity;
import com.rocoinfo.rocomall.enumconst.CacheableTag;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.utils.ReflectionUtils;

/**
 * <dl>
 * <dd>描述:service基础类--获取和更新</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：15/7/30 上午11:50</dd>
 * <dd>创建人： weiys</dd>
 * </dl>
 */
@SuppressWarnings("all")
public abstract class CrudService<D extends CrudDao<T>, T extends IdEntity> extends BaseService<T> {

	/**
	 * 持久层对象
	 */
	@Autowired
	protected D entityDao;
	@Autowired
	protected RedisCacheManager cacheManager;

	@Override
	public T getById(Long id) {
		if (id == null || id < 1)
			return null;
		return getAndSetToCacheIfRequired(id);
	}

	@Override
	public void insert(T entity) {
		if (entity == null)
			return;
		entityDao.insert(entity);
		evictCacheEntitysHook();
	}

	protected void evictCacheEntitysHook() {
	}

	@Override
	public void update(T entity) {
		if (entity == null)
			return;
		entityDao.update(entity);
		evictCachedEntityIfRequired(entity.getId());
	}

	@Override
	public void deleteById(Long id) {
		if (id == null || id < 1)
			return;
		this.entityDao.deleteById(id);
		evictCachedEntityIfRequired(id);
	}

	@Override
	public List<T> findAll() {
		return this.entityDao.findAll();
	}

	@Override
	public Page<T> searchScrollPage(Map<String, Object> params, Pageable pageable) {
		params.put(Constants.PAGE_OFFSET, pageable.getPageNumber());
		params.put(Constants.PAGE_SIZE, pageable.getPageSize());
		params.put(Constants.PAGE_SORT, pageable.getSort());

		List<T> pageData = Collections.emptyList();
		Long count = this.entityDao.searchTotal(params);
		if (count > 0) {
			pageData = entityDao.search(params);
		}
		return new PageImpl<T>(pageData, pageable, count);
	}

	private T getAndSetToCacheIfRequired(Long id) {
		T entity = null;
		CacheableTag cacheTag = getEntityCachableAnnotation();
		if (cacheTag != null && cacheTag.isCacheRequired()) {
			String key = cacheTag.cacheKeyPrev() + id;
			Cache cache = cacheManager.getCache(CacheKeys.DEFAULT_NAME);
			ValueWrapper cachedEntity = cache.get(key);
			if (cachedEntity != null) {
				entity = (T) cachedEntity.get();
			} else {
				entity = entityDao.getById(id);
				if (entity != null) {
					cache.put(key, entity);
				}
			}
		} else {
			entity = entityDao.getById(id);
		}
		return entity;
	}

	private void evictCachedEntityIfRequired(Long id) {
		CacheableTag cacheTag = getEntityCachableAnnotation();
		if (cacheTag != null && cacheTag.isCacheRequired()) {
			String key = cacheTag.cacheKeyPrev() + id;
			Cache cache = cacheManager.getCache(CacheKeys.DEFAULT_NAME);
			cache.evict(key);
		}
		evictCacheEntitysHook();
	}

	private CacheableTag getEntityCachableAnnotation() {
		Class<T> cls = ReflectionUtils.getSuperClassGenricType(this.getClass(), 1);
		if (cls.isAnnotationPresent(CacheableTag.class)) {
			CacheableTag cacheTag = cls.getAnnotation(CacheableTag.class);
			return cacheTag;
		}
		return null;
	}
}
