package com.rocoinfo.rocomall.service.impl.wx;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.rocoinfo.rocomall.Constants;
import com.rocoinfo.rocomall.WeiXinInterfaceUrl;
import com.rocoinfo.rocomall.dto.StatusDto;
import com.rocoinfo.rocomall.entity.wx.Menu;
import com.rocoinfo.rocomall.entity.wx.WXUser;
import com.rocoinfo.rocomall.entity.wx.WxMenu;
import com.rocoinfo.rocomall.enumconst.Gender;
import com.rocoinfo.rocomall.redis.CacheKeys;
import com.rocoinfo.rocomall.redis.JedisTemplate;
import com.rocoinfo.rocomall.service.wx.IWxApiService;
import com.rocoinfo.rocomall.utils.HttpUtils;
import com.rocoinfo.rocomall.utils.JsonUtils;

/**
 * <dl>
 * <dd>描述: 访问微信公共平台 接口api Service</dd>
 * <dd>公司: 大城若谷信息技术有限公司</dd>
 * <dd>创建时间：2016年3月1日 上午11:10:50</dd>
 * <dd>创建人： 张敏</dd>
 * </dl>
 */
@Service
public class WxApiService implements IWxApiService {
	private static final int EXPIRE_SECORD = 110 * 60;
	@Autowired
	private JedisTemplate jedisTemplate;
	@Autowired
	private WxMenuService menuService;

	/**
	 * 获取访问token http GET请求方式: 获取access_token 请求地址 access_token有效期为7200秒，重复获取将导致上次获取的access_token失效
	 * 公众号可以使用AppID和AppSecret调用本接口来获取access_token
	 * 
	 * 正常情况下，微信会返回下述JSON数据包给公众号： {"access_token":
	 * "WDk2fDLcMe_hz63v5iCwM0dDJvDK9h7A1QtBCggCm0A_swd_BD3Ps4Tl-XqiXQ92dYS8wEx_qeoc5aPTuoZ_eQk5BrbOuA34mlUHZNRkcDUzQZHpLBr2KgJZCtEDLn0GDSo0eKkofhmJbXavbdyYeg"
	 * ,"expires_in":7200}
	 */
	public String getAccessToken(String appId, String appSecret) {
		String cachedToken = jedisTemplate.get(CacheKeys.WX_ACCES_TOKEN_REDIS_KEY);
		if (StringUtils.isNotEmpty(cachedToken)) {
			return cachedToken;
		} else {
			String accessToken = StringUtils.EMPTY;
			String getAccessTokenUrl = WeiXinInterfaceUrl.GET_ACCESS_TOKEN;
			getAccessTokenUrl = String.format(getAccessTokenUrl, appId, appSecret);
			String respJson = HttpUtils.get(getAccessTokenUrl);
			Map map = JsonUtils.fromJson(respJson, Map.class);
			if (map != null) {
				accessToken = StringUtils.trimToEmpty((String) map.get("access_token"));
				if (StringUtils.isNotEmpty(accessToken)) {
					jedisTemplate.setex(CacheKeys.WX_ACCES_TOKEN_REDIS_KEY, accessToken, EXPIRE_SECORD);
				}
			}
			return accessToken;
		}
	}

	/***
	 * 创建微信自定义菜单，返回创建结果消息 正确时的返回JSON数据包如下： {"errcode":0,"errmsg":"ok"} 错误时的返回JSON数据包如下（示例为无效菜单名长度）：
	 * {"errcode":40018,"errmsg":"invalid button name size"}
	 */
	public StatusDto<String> createWeixinMenu(String appId, String appSecret) {

		List<Menu> topMenuList = menuService.findTopWithSubMenus();

		if (CollectionUtils.isNotEmpty(topMenuList)) {
			List<WxMenu> createMenuList = new ArrayList<WxMenu>(topMenuList.size());

			for (Menu topMenu : topMenuList) {
				WxMenu createdWxMenu = WxMenu.valueOf(topMenu);

				//如果有子菜单
				List<Menu> subMenuList = topMenu.getChildMenuList();
				if (subMenuList != null && !subMenuList.isEmpty()) {
					for (Menu sub : subMenuList) {
						createdWxMenu.addSubMenu(WxMenu.valueOf(sub));
					}
				}
				createMenuList.add(createdWxMenu);
			}

			String postUrl = WeiXinInterfaceUrl.WX_MENU_CREATE;
			postUrl = String.format(postUrl, getAccessToken(appId, appSecret));

			Map<String, List<WxMenu>> map = new HashMap<String, List<WxMenu>>();
			map.put("button", createMenuList);
			String reqJson = JsonUtils.pojoToJson(map);

			String respJson = HttpUtils.postJson(postUrl, reqJson);
			Map resultMap = JsonUtils.fromJson(respJson, Map.class);
			if (resultMap != null) {

				StatusDto<String> respMsg = new StatusDto<String>();
				String resultCode = String.valueOf(resultMap.get("errcode"));
				if ("0".equals(resultCode)) {
					//"0".equals(respMsg.getCode()  ==0表示创建菜单成功
					respMsg.setCode(Constants.RESP_STATUS_CODE_SUCCESS);
				} else {
					respMsg.setCode(resultCode);
				}
				respMsg.setMessage((String) resultMap.get("errmsg"));
				return respMsg;
			} else {
				return StatusDto.buildFailureStatusDto("创建微信菜单 接口发生异常！");
			}
		} else {
			return StatusDto.buildSuccessStatusDto("后台没有配置任何菜单，不能创建");
		}
	}

	/**
	 * 到微信公共平台，拉取微信用户更多的基本信息,没有拉取到则返回null对象
	 * 
	 * @param openId
	 */
	@Override
	public WXUser getWxUserInfo(final String appId, final String appSecret, final String openId) {
		WXUser wxUser = null;
		if (StringUtils.isNotBlank(openId)) {
			String getUserInfoUrl = WeiXinInterfaceUrl.WX_USER_INFO;
			getUserInfoUrl = String.format(getUserInfoUrl, getAccessToken(appId, appSecret), openId);
			String respJson = HttpUtils.get(getUserInfoUrl);
			Map map = JsonUtils.fromJson(respJson, Map.class);
			if (map != null && NumberUtils.toInt(String.valueOf(map.get("subscribe"))) == 1) {
				wxUser = new WXUser();
				wxUser.setNickName((String) map.get("nickname"));
				//用户的性别，值为1时是男性，值为2时是女性，值为0时是未知
				Integer sex = (Integer) map.get("sex");
				wxUser.setSex(sex == 1 ? Gender.male : sex == 2 ? Gender.female : Gender.unknown);
				wxUser.setCountry((String) map.get("country"));
				wxUser.setProvince((String) map.get("province"));
				wxUser.setCity((String) map.get("city"));
				wxUser.setHeadImgUrl((String) map.get("headimgurl"));
			}
		}
		return wxUser;
	}

	/***
	 * 人工主动发送客服消息【下发回复/群发push】 http请求方式: POST
	 */
	//public <T extends SendMsg> void sendMessage(T sendMsg);

	/**
	 * 微信下载多媒体文件
	 * 
	 * @param mediaId 要下载的文件 mediaId号
	 * @param saveRealDir 下载下来的多媒体文件 保存的目录
	 */
	public void downloadMediaFile(final String mediaId, final String saveRealDir) {

	}

	// =============  微信公共平台jsSdk接口  ==============

	/**
	 * 生成签名之前必须先了解一下jsapi_ticket，jsapi_ticket是公众号用于调用微信JS接口的临时票据。
	 * 正常情况下，jsapi_ticket的有效期为7200秒，通过access_token来获取。由于获取jsapi_ticket的api调用次数非常有限
	 * ，频繁刷新jsapi_ticket会导致api调用受限，影响自身业务，开发者必须在自己的服务全局缓存jsapi_ticket <br/>
	 * 成功返回如下JSON：<br/>
	 * { "errcode":0, "errmsg":"ok",
	 * "ticket":"bxLdikRXVbTPdHSM05e5u5sUoXNKd8-41ZO3MhKoyN5OfkWITDGgnr2fwJ0m9E8NYzWKVZvdVtaUgWvsdshFKA",
	 * "expires_in":7200 } <br/>
	 * 获得jsapi_ticket之后，就可以生成JS-SDK权限验证的签名了
	 */
	public String getJsApiTicket(String appId, String appSecret) {

		String cachedJsApiTicket = jedisTemplate.get(CacheKeys.WX_JS_API_TICKET_REDIS_KEY);
		if (StringUtils.isNotEmpty(cachedJsApiTicket)) {
			return cachedJsApiTicket;
		} else {
			String jsApiTicket = StringUtils.EMPTY;
			String jsApiTicketUrl = WeiXinInterfaceUrl.JSAPI_TICKET_URL;
			jsApiTicketUrl = String.format(jsApiTicketUrl, getAccessToken(appId, appSecret));
			String respJson = HttpUtils.get(jsApiTicketUrl);
			Map map = JsonUtils.fromJson(respJson, Map.class);
			if (map != null) {
				jsApiTicket = StringUtils.trimToEmpty((String) map.get("ticket"));
				if (StringUtils.isNotEmpty(jsApiTicket)) {
					// 110分钟
					jedisTemplate.setex(CacheKeys.WX_JS_API_TICKET_REDIS_KEY, jsApiTicket, EXPIRE_SECORD);
				}
			}
			return jsApiTicket;
		}
	}

}
