package com.rocoinfo.rocomall.utils;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import com.rocoinfo.rocomall.common.service.ExcelFormatException;
import com.rocoinfo.rocomall.common.service.ServiceException;

/**
 * 该类实现了将一组对象转换为Excel表格，并且可以从Excel表格中读取到一组List对象中 该类利用了BeanUtils框架中的反射完成 使用该类的前提，在相应的实体对象上通过@ExcelTitle来完成相应的注解
 * 
 * @author zhangmin
 */
public final class ExcelUtil {

	private static ExcelUtil util = new ExcelUtil();

	private ExcelUtil() {
	}

	public static ExcelUtil getInstance() {
		return util;
	}

	/**
	 * 处理对象转换为Excel
	 * 
	 * @param datas
	 * @param excelTemplateFile
	 * @param exportDataList
	 * @param clz
	 * @param isClassPath
	 */
	private <E> ExcelTemplate handleObj2Excel(final String excelTemplateFile, List<E> exportDataList, Class<E> clz, boolean isClassPath) {
		ExcelTemplate template = new ExcelTemplate();

		if (isClassPath) {
			template.readTemplateByClasspath(excelTemplateFile);
		} else {
			template.readTemplateByPath(excelTemplateFile);
		}
		List<ExcelHeader> headers = getHeaderList(clz);
		// 输出标题
		template.createNewRow();
		for (ExcelHeader eh : headers) {
			template.createCell(eh.getTitle());
		}

		// 写数据到Excel
		for (E rowData : exportDataList) {
			template.createNewRow();
			for (ExcelHeader eh : headers) {
				createCellAndWriteValue(template, rowData, eh);
			}
		}
		return template;
	}

	private <E> void createCellAndWriteValue(ExcelTemplate template, E rowData, ExcelHeader excelHeader) {
		Object propValue = ReflectionUtils.getFieldValue(rowData, excelHeader.getPropertyName());
		if (propValue == null) {
			propValue = StringUtils.EMPTY;
		}

		if (propValue instanceof Date) {
			template.createCell((Date) propValue);
		} else if (propValue instanceof Double) {
			template.createCell((Double) propValue);
		} else if (propValue instanceof Integer) {
			template.createCell((Integer) propValue);
		} else if (propValue instanceof Boolean) {
			template.createCell((Boolean) propValue);
		} else {
			template.createCell(propValue.toString());
		}
	}

	/**
	 * 将对象转换为Excel并且导出，该方法是基于模板的导出，导出到流
	 * 
	 * @param variableExtras 模板中的替换的变量数据
	 * @param excelTemplatePath 模板路径
	 * @param os 输出流
	 * @param exportDataList 对象列表
	 * @param clz 对象的类型
	 * @param isClasspath 模板是否在classPath路径下
	 */
	public <E> void exportObj2ExcelByTemplate(Map<String, String> variableExtras, String excelTemplatePath, OutputStream os, List<E> exportDataList, Class<E> clz, boolean isClasspath) {
		ExcelTemplate et = handleObj2Excel(excelTemplatePath, exportDataList, clz, isClasspath);
		et.replaceFinalData(variableExtras);
		et.wirteToStream(os);
	}

	/**
	 * 将对象转换为Excel并且导出，该方法是基于模板的导出，导出到一个具体的路径中
	 * 
	 * @param variableExtras 模板中的替换的常量数据
	 * @param excelTemplatePath 模板路径
	 * @param outPath 输出路径
	 * @param exportDataList 对象列表
	 * @param clz 对象的类型
	 * @param isClasspath 模板是否在classPath路径下
	 */
	public <E> void exportObj2ExcelByTemplate(Map<String, String> variableExtras, String excelTemplatePath, String outPath, List<E> exportDataList, Class<E> clz, boolean isClasspath) {
		ExcelTemplate et = handleObj2Excel(excelTemplatePath, exportDataList, clz, isClasspath);
		et.replaceFinalData(variableExtras);
		et.writeToFile(outPath);
	}

	/**
	 * 将对象转换为Excel并且导出，该方法是基于模板的导出，导出到流,基于Properties作为常量数据
	 * 
	 * @param variableExtras 基于Properties的模板变量数据
	 * @param excelTemplatePath 模板路径
	 * @param os 输出流
	 * @param exportDataList 对象列表
	 * @param clz 对象的类型
	 * @param isClasspath 模板是否在classPath路径下
	 */
	public <E> void exportObj2ExcelByTemplate(Properties variableExtras, String excelTemplatePath, OutputStream os, List<E> exportDataList, Class<E> clz, boolean isClasspath) {
		ExcelTemplate et = handleObj2Excel(excelTemplatePath, exportDataList, clz, isClasspath);
		et.replaceFinalData(variableExtras);
		et.wirteToStream(os);
	}

	/**
	 * 将对象转换为Excel并且导出，该方法是基于模板的导出，导出到一个具体的路径中,基于Properties作为常量数据
	 * 
	 * @param variableExtras 基于Properties的常量数据模型
	 * @param excelTemplatePath 模板路径
	 * @param outFile 输出路径
	 * @param exportDataList 对象列表
	 * @param clz 对象的类型
	 * @param isClasspath 模板是否在classPath路径下
	 */
	public <E> void exportObj2ExcelByTemplate(Properties variableExtras, String excelTemplatePath, String outFile, List<E> exportDataList, Class<E> clz, boolean isClasspath) {
		ExcelTemplate et = handleObj2Excel(excelTemplatePath, exportDataList, clz, isClasspath);
		et.replaceFinalData(variableExtras);
		et.writeToFile(outFile);
	}

	private <E> Workbook handleObj2Excel(List<E> exportDataList, Class<E> clz, boolean isXssf, ExcelMergeRegion... mergeCells) {
		Workbook wb = null;
		if (isXssf) {
			wb = new XSSFWorkbook();
		} else {
			wb = new HSSFWorkbook();
		}
		Sheet sheet = wb.createSheet();

		Row row = sheet.createRow(0);
		List<ExcelHeader> headers = getHeaderList(clz);
		// 写标题
		for (int i = 0; i < headers.size(); i++) {
			row.createCell(i).setCellValue(headers.get(i).getTitle());
		}
		// 写数据
		Object obj = null;
		for (int rowIdx = 0; rowIdx < exportDataList.size(); rowIdx++) {
			row = sheet.createRow(rowIdx + 1);
			obj = exportDataList.get(rowIdx);
			for (int colIdx = 0; colIdx < headers.size(); colIdx++) {
				Cell cell = row.createCell(colIdx);
				CellStyle style = wb.createCellStyle();
				style.setAlignment(CellStyle.ALIGN_CENTER);
				style.setVerticalAlignment(CellStyle.VERTICAL_CENTER);
				style.setBorderTop(CellStyle.BORDER_THIN);
				style.setBorderLeft(CellStyle.BORDER_THIN);
				cell.setCellStyle(style);

				Object propValue = ReflectionUtils.getFieldValue(obj, headers.get(colIdx).getPropertyName());
				if (propValue == null) {
					propValue = StringUtils.EMPTY;
				}

				if (propValue instanceof Date) {
					cell.setCellValue((Date) propValue);
				} else if (propValue instanceof Double) {
					cell.setCellValue((Double) propValue);
				} else if (propValue instanceof Integer) {
					cell.setCellValue((Integer) propValue);
				} else if (propValue instanceof Boolean) {
					cell.setCellValue((Boolean) propValue);
				} else {
					cell.setCellValue(propValue.toString());
				}
			}
		}

		if (exportDataList.size() > 0 && ArrayUtils.isNotEmpty(mergeCells)) {
			for (ExcelMergeRegion range : mergeCells) {
				sheet.addMergedRegion(new CellRangeAddress(range.getFirstRow(), range.getLastRow(), range.getFirstCol(), range.getLastCol()));
			}
		}
		return wb;
	}

	/**
	 * 导出对象到Excel，不是基于模板的，直接新建一个Excel完成导出，基于路径的导出
	 * 
	 * @param outFileAbsolutePath 导出路径
	 * @param exportDataList 导出的数据列表
	 * @param clz 对象类型
	 * @param isXssf 是否是2007版本
	 */
	public <E> void exportObj2Excel(String outFileAbsolutePath, List<E> exportDataList, Class<E> clz, boolean isXssf, ExcelMergeRegion... mergeCells) {
		Workbook wb = handleObj2Excel(exportDataList, clz, isXssf, mergeCells);
		FileOutputStream fos = null;
		try {
			fos = new FileOutputStream(outFileAbsolutePath);
			wb.write(fos);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				if (fos != null)
					fos.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * 导出对象到Excel，不是基于模板的，直接新建一个Excel完成导出，基于流
	 * 
	 * @param os 输出流
	 * @param exportDataList 对象列表
	 * @param clz 对象类型
	 * @param isXssf 是否是2007版本
	 */
	public <E> void exportObj2Excel(OutputStream os, List<E> exportDataList, Class<E> clz, boolean isXssf, ExcelMergeRegion... mergeCells) {
		try {
			Workbook wb = handleObj2Excel(exportDataList, clz, isXssf, mergeCells);
			wb.write(os);
			os.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 从类路径读取相应的Excel文件到对象列表
	 * 
	 * @param clsPath 类路径下的path
	 * @param clz 对象类型
	 * @param titleRowIndex 开始行，注意是标题所在行
	 */
	public <E> List<E> readExcel2ObjsByClasspath(String clsPath, Class<E> clz, int titleRowIndex) {
		Workbook wb = null;
		try {
			wb = WorkbookFactory.create(getClass().getResourceAsStream(clsPath));
			return handleExcel2Objs(wb, clz, titleRowIndex);
		} catch (InvalidFormatException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 从文件路径读取相应的Excel文件到对象列表
	 * 
	 * @param path 文件路径下的path
	 * @param clz 对象类型
	 */
	public <E> List<E> readExcel2ObjsByStream(InputStream inputStream, Class<E> clz) {
		Workbook wb = null;
		try {
			wb = WorkbookFactory.create(inputStream);
			return handleExcel2Objs(wb, clz, 0);
		} catch (InvalidFormatException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return Collections.emptyList();
	}

	/**
	 * 这种方式更加节省内存开销
	 * 
	 * @param file ecxel 文件对象
	 * @param clz
	 * @return
	 * @author： 张文山
	 * @创建时间：2015-9-10 下午1:49:28
	 */
	public <E> List<E> readExcel2ObjsByFile(File file, Class<E> clz) {
		Workbook wb = null;
		try {
			wb = WorkbookFactory.create(file);
			return handleExcel2Objs(wb, clz, 0);
		} catch (InvalidFormatException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return Collections.emptyList();
	}

	/**
	 * 从文件路径读取相应的Excel文件到对象列表
	 * 
	 * @param path 文件路径下的path
	 * @param clz 对象类型
	 * @param titleRowIndex 开始行，注意是标题所在行
	 */
	public <E> List<E> readExcel2ObjsByPath(String path, Class<E> clz, int titleRowIndex) {
		Workbook wb = null;
		try {
			wb = WorkbookFactory.create(new File(path));
			return handleExcel2Objs(wb, clz, titleRowIndex);
		} catch (InvalidFormatException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 从类路径读取相应的Excel文件到对象列表，标题行为0，没有尾行
	 * 
	 * @param clsPath 路径
	 * @param clz 类型
	 * @return 对象列表
	 */
	public <E> List<E> readExcel2ObjsByClasspath(String clsPath, Class<E> clz) {
		return this.readExcel2ObjsByClasspath(clsPath, clz, 0);
	}

	/**
	 * 从文件路径读取相应的Excel文件到对象列表，标题行为0，没有尾行
	 * 
	 * @param path 路径
	 * @param clz 类型
	 * @return 对象列表
	 */
	public <E> List<E> readExcel2ObjsByPath(String path, Class<E> clz) {
		return this.readExcel2ObjsByPath(path, clz, 0);
	}

	private Object getCellValue(Cell cell) {
		Object cellValue = null;
		switch (cell.getCellType()) {
		case Cell.CELL_TYPE_BLANK:
			cellValue = "";
			break;
		case Cell.CELL_TYPE_BOOLEAN:
			cellValue = cell.getBooleanCellValue();
			break;
		case Cell.CELL_TYPE_FORMULA:
			cellValue = cell.getCellFormula();
			break;
		case Cell.CELL_TYPE_NUMERIC:
			// 日期
			if (DateUtil.isCellDateFormatted(cell)) {
				cellValue = cell.getDateCellValue();
			} else {
				// 其他数字类型也按照字符串处理，因为浮点数可能用科学计数法表示
				cellValue = new DecimalFormat("#").format(cell.getNumericCellValue());
			}
			break;
		case Cell.CELL_TYPE_STRING:
			cellValue = cell.getStringCellValue();
			break;
		default:
			cellValue = null;
			break;
		}
		return cellValue;
	}

	private <E> List<E> handleExcel2Objs(Workbook wb, Class<E> clz, int titleRowIndex) {
		Sheet sheet = wb.getSheetAt(0);
		if (sheet.getLastRowNum() < 1) {
			throw new RuntimeException("excel表格没有任何数据!");
		}
		Row row = sheet.getRow(titleRowIndex);
		List<E> objs = new ArrayList<E>();
		Map<Integer, ExcelHeader> colIndexPropertyMap = getHeaderMap(row, clz);

		if (colIndexPropertyMap == null || colIndexPropertyMap.isEmpty()) {
			throw new RuntimeException("要读取的Excel的格式不正确，检查是否设定了合适的行");
		}
		for (int dataRowIdx = titleRowIndex + 1; dataRowIdx <= sheet.getLastRowNum(); dataRowIdx++) {
			row = sheet.getRow(dataRowIdx);
			E obj = (E) getInstance(clz);
			for (int colIndx = 0; colIndx < row.getLastCellNum(); colIndx++) {
				Cell cell = row.getCell(colIndx);
				ExcelHeader header = colIndexPropertyMap.get(colIndx);
				if (header == null) {
					continue;
				}

				String errorMsg = "第%d行,第%d列为必填项.";
				Object cellValue = this.getCellValue(cell);
				if (header.isRequired() && (cellValue == null || (cellValue instanceof String && StringUtils.isBlank((String) cellValue)))) {
					String error = String.format(errorMsg, dataRowIdx + 1, colIndx + 1);
					throw new ExcelFormatException(error);
				}

				if (cellValue != null) {
					try {
						ReflectionUtils.setFieldValue(obj, header.getPropertyName(), cellValue);
					} catch (Exception e) {
						StringBuffer sb = new StringBuffer("第");
						sb.append(dataRowIdx + 1).append("行,第").append(colIndx + 1).append("列数据【").append(cellValue).append("】格式不正确");
						if (StringUtils.isNotBlank(header.getDataFormat())) {
							sb.append(",正确格式【").append(header.getDataFormat()).append("】.");
						}
						throw new ExcelFormatException(sb.toString());
					}
				}

			}
			objs.add(obj);
		}
		return objs;
	}

	private <E> E getInstance(Class<E> clz) {
		try {
			return clz.newInstance();
		} catch (InstantiationException e) {
			e.printStackTrace();
			throw new ServiceException("创建对象失败");
		} catch (IllegalAccessException e) {
			e.printStackTrace();
			throw new ServiceException("创建对象失败");
		}
	}

	private <E> List<ExcelHeader> getHeaderList(Class<E> clz) {
		List<ExcelHeader> headers = new ArrayList<ExcelHeader>();
		// 在 Java Bean 上进行内省，了解其所有属性、公开的方法和事件。
		try {
			BeanInfo info = Introspector.getBeanInfo(clz);
			PropertyDescriptor[] proertyDescritors = info.getPropertyDescriptors();
			for (PropertyDescriptor pd : proertyDescritors) {
				String propertyName = pd.getName();
				Method method = pd.getReadMethod();
				if (method.isAnnotationPresent(ExcelTitle.class)) {
					ExcelTitle excelTitle = method.getAnnotation(ExcelTitle.class);
					headers.add(new ExcelHeader(excelTitle.title(), excelTitle.order(), propertyName, excelTitle.required(), excelTitle.dataFormat()));
				}
			}
		} catch (IntrospectionException e) {
			e.printStackTrace();
		}

		Collections.sort(headers);
		return headers;
	}

	private <E> Map<Integer, ExcelHeader> getHeaderMap(Row titleRow, Class<E> clz) {
		List<ExcelHeader> headers = getHeaderList(clz);
		Map<Integer, ExcelHeader> maps = new HashMap<Integer, ExcelHeader>();
		int cessCounts = 0;
		for (Cell cell : titleRow) {
			String title = cell.getStringCellValue();
			if (!makeLinks(headers, cell, maps)) {
				throw new ExcelFormatException("列名【" + title + "】无法匹配,请下载最新模板!");
			}
			cessCounts++;
		}
		if (cessCounts != headers.size()) {
			throw new ExcelFormatException("缺少数据列,请下载最新模板!");
		}
		return maps;
	}

	private boolean makeLinks(List<ExcelHeader> headers, Cell cell, Map<Integer, ExcelHeader> maps) {
		for (ExcelHeader eh : headers) {
			if (eh.getTitle().equals(cell.getStringCellValue().trim())) {
				maps.put(cell.getColumnIndex(), eh);
				return true;
			}
		}
		return false;
	}
}