package com.rocoinfo.weixin.api;

import com.rocoinfo.weixin.model.ApiResult;
import com.rocoinfo.weixin.util.HttpUtils;
import com.rocoinfo.weixin.util.JsonUtils;
import com.rocoinfo.weixin.util.MapUtils;

import java.util.List;
import java.util.Map;

/**
 * <dl>
 * <dd>Description: 用户组管理接口</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/3/6 上午10:58</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public class GroupApi extends BaseApi {

    /**
     * 创建分组请求url
     */
    private static final String CREATE_GROUP_URL = "https://api.weixin.qq.com/cgi-bin/groups/create?access_token=";

    /**
     * 创建分组参数
     */
    private static final String CREATE_PARAMS_JSON = "{\"group\":{\"name\":\"%s\"}}";

    private GroupApi() {
        super();
    }

    /**
     * 创建用户分组
     *
     * @param name 分组名字（30个字符以内）
     * @return {"group": {"id": 107,"name": "test"}}
     */
    public static ApiResult create(final String name) {
        String url = CREATE_GROUP_URL + getAccessToken();
        return ApiResult.build(HttpUtils.post(url, String.format(CREATE_PARAMS_JSON, name)));
    }

    /**
     * 获取所有用户组的url
     */
    private static final String GET_URL = "https://api.weixin.qq.com/cgi-bin/groups/get?access_token=";

    /**
     * 查询所有用户组
     *
     * @return 返回用户组列表信息 eg: {"groups"[{"id": 1,"name": "好友","count": 344},...]}
     */
    public static ApiResult all() {
        String url = GET_URL + getAccessToken();
        return ApiResult.build(HttpUtils.get(url));
    }

    /**
     * 查询用户所在分组id的url
     */
    private static final String USER_PLACE_GROUPID_URL = "https://api.weixin.qq.com/cgi-bin/groups/getid?access_token=";

    /**
     * 获取用户所在分组id
     *
     * @param openid 用户openid
     * @return 返回用户分组id eg: {"groupid": 102}
     */
    public static ApiResult userGroup(String openid) {
        Map<String, String> params = MapUtils.of("openid", openid);
        String url = USER_PLACE_GROUPID_URL + getAccessToken();
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }

    /**
     * 修改分组名称url
     */
    private static final String UPDATE_GROUP_URL = "https://api.weixin.qq.com/cgi-bin/groups/update?access_token=";

    /**
     * 修改分组名称 post请求格式: {"group": {"id": 107,"name": "test123"}}
     *
     * @param id   分组id
     * @param name 分组名称
     * @return 返回修改结果
     */
    public static ApiResult update(String id, String name) {
        Map<String, String> group = MapUtils.of("id", id, "name", name);
        Map<String, Map<String, String>> params = MapUtils.of("group", group);

        String url = UPDATE_GROUP_URL + getAccessToken();
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }

    /**
     * 移动用户请求url
     */
    private static final String MOVE_MEMBER_URL = "https://api.weixin.qq.com/cgi-bin/groups/members/update?access_token=";

    /**
     * 移动指定用户到指定组
     *
     * @param openid  移动用户openid
     * @param groupid 需要移动的组id
     * @return 返回移动信息
     */
    public static ApiResult move(String openid, String groupid) {
        Map<String, String> params = MapUtils.of("openid", openid, "to_groupid", groupid);
        String url = MOVE_MEMBER_URL + getAccessToken();
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }

    /**
     * 批量移动用户的url
     */
    private static final String BATCH_MOVE_MEMBER_URL = "https://api.weixin.qq.com/cgi-bin/groups/members/batchupdate?access_token=";

    /**
     * 批量移动用户到指定分组
     *
     * @param openids 用户openids
     * @param groupid 移动到组的id
     * @return 返回移动结果
     */
    public static ApiResult batchMoveMember(List<String> openids, String groupid) {
        Map<String, Object> params = MapUtils.of("openid_list", openids, "to_groupid", groupid);
        String url = BATCH_MOVE_MEMBER_URL + getAccessToken();
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }

    /**
     * 删除指定分组的url
     */
    private static final String DELETE_GROUP_URL = "https://api.weixin.qq.com/cgi-bin/groups/delete?access_token=";

    /**
     * 删除指定分组
     *
     * @param id 分组id
     * @return 返回删除结果
     */
    public static ApiResult delete(final String id) {
        Map<String, String> group = MapUtils.of("id", id);
        Map<String, Map<String, String>> params = MapUtils.of("group", group);
        String url = DELETE_GROUP_URL + getAccessToken();
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }
}
