package com.rocoinfo.weixin.api;

import com.rocoinfo.weixin.config.ParamManager;
import com.rocoinfo.weixin.model.ApiResult;
import com.rocoinfo.weixin.model.JssdkResult;
import com.rocoinfo.weixin.ticket.TicketManager;
import com.rocoinfo.weixin.util.HttpUtils;
import com.rocoinfo.weixin.util.JssdkUtils;
import com.rocoinfo.weixin.util.SignUtils;
import com.rocoinfo.weixin.util.StringUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Formatter;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * <dl>
 * <dd>Description: 获取jsApiTicket的接口</dd>
 * <dd>@date：2017/8/16 17:06</dd>
 * <dd>@author：Paul</dd>
 * </dl>
 */
public class JsApiTicketApi extends BaseApi {

    /**
     * 获取jsApiTicket的 url
     */
    private static final String JSAPI_TICKET = "https://api.weixin.qq.com/cgi-bin/ticket/getticket?access_token=%s&type=jsapi";

    private JsApiTicketApi() {
        super();
    }

    /**
     * 获取 jsapi_ticket,从远程调用获取
     *
     * @return
     */
    public static ApiResult getTicket() {
        String url = String.format(JSAPI_TICKET, getAccessToken());
        return ApiResult.build(HttpUtils.get(url));
    }

    /**
     * 获取字符串格式的 jsapi_ticket
     *
     * @return
     */
    public static String getTicketStr() {
        ApiResult res = getTicket();
        if (res.isSuccess()) {
            return String.valueOf(res.get("ticket"));
        }
        return null;
    }

    /**
     * 根据url进行jssdk的签名校验，并将结果封装成jssdk需要的格式
     * 如果签名失败返回null
     *
     * @param url url
     * @return
     */
    public static JssdkResult signature(String url) throws UnsupportedEncodingException {
        // 处理url链接中的#号,因为签名的时候url中不包括#号部分
        if (url.contains("#")) {
            url = url.substring(0, url.indexOf("#"));
        }
        //生成随机数和时间戳
        String nonceStr = JssdkUtils.generateNonceStr(16);
        String timestamp = JssdkUtils.generateTimestamp();
        // jssdk
        String tickect = TicketManager.getJsApiTicket();
        // 签名
        String signature;

        try {
            //构建待加密字符串
            String encryptStr = "jsapi_ticket=" + tickect +
                    "&noncestr=" + nonceStr +
                    "&timestamp=" + timestamp +
                    "&url=" + URLDecoder.decode(url, "utf-8");

            signature = JssdkUtils.encryptMessageBySHA1(encryptStr);
        } catch (Exception e) {
            return null;
        }

        // 签名不能为空
        if (StringUtils.isBlank(signature)) {
            return null;
        }

        return new JssdkResult().setAppId(ParamManager.getAppid())
                .setNonceStr(nonceStr)
                .setTimestamp(timestamp)
                .setSignature(signature);
    }
}
