package com.rocoinfo.weixin.api;

import com.rocoinfo.weixin.enums.MediaType;
import com.rocoinfo.weixin.model.ApiResult;
import com.rocoinfo.weixin.model.MediaArticle;
import com.rocoinfo.weixin.model.MediaFile;
import com.rocoinfo.weixin.util.HttpUtils;
import com.rocoinfo.weixin.util.JsonUtils;
import com.rocoinfo.weixin.util.MapUtils;

import java.io.File;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 素材管理，没时间写，copy自jfinal-weixin
 *
 * @author l.cm
 *         文档：http://mp.weixin.qq.com/wiki/5/963fc70b80dc75483a271298a76a8d59.html
 * @author aaron 重构
 */
public class MediaApi extends BaseApi {

    private static final String UPLOAD_URL = "https://api.weixin.qq.com/cgi-bin/media/upload?access_token=";

    /**
     * 上传临时素材
     *
     * @param type 上传的临时多媒体文件有格式
     * @param file 需要上传的文件
     * @return
     */
    public static ApiResult upload(MediaType type, File file) {
        String url = UPLOAD_URL + getAccessToken() + "&type=" + type.getValue();
        return ApiResult.build(HttpUtils.upload(url, file, null));
    }

    private static final String GET_URL = "https://api.weixin.qq.com/cgi-bin/media/get?access_token=";

    /**
     * 获取临时素材
     *
     * @param mediaId 素材Id
     * @return
     */
    public static MediaFile get(String mediaId) {
        String url = GET_URL + getAccessToken() + "&media_id=" + mediaId;
        return HttpUtils.download(url);
    }

    private static final String ADD_NEWS_URL = "https://api.weixin.qq.com/cgi-bin/material/add_news?access_token=";

    /**
     * 新增永久图文素材
     *
     * @param articles 图文素材列表
     * @return
     */
    public static ApiResult addNews(List<MediaArticle> articles) {
        String url = ADD_NEWS_URL + getAccessToken();
        Map<String, List<MediaArticle>> dataMap = MapUtils.of("articles", articles);
        String jsonResult = HttpUtils.post(url, JsonUtils.toJson(dataMap));
        return ApiResult.build(jsonResult);
    }

    private static final String ADD_MATERIAL_URL = "https://api.weixin.qq.com/cgi-bin/material/add_material?access_token=";

    /**
     * 新增其他类型永久素材
     *
     * @param file 文件
     * @return ApiResult
     */
    public static ApiResult addMaterial(File file) {
        String url = ADD_MATERIAL_URL + getAccessToken();
        return ApiResult.build(HttpUtils.upload(url, file, null));
    }

    private static final String UPLOAD_VIDEO_URL = "https://file.api.weixin.qq.com/cgi-bin/media/uploadvideo?access_token=";

    /**
     * 视频群发的消息素材上传
     *
     * @param mediaId     用于群发的消息的media_id
     * @param title       消息的标题
     * @param description 消息的描述
     * @return
     */
    public static ApiResult uploadVideo(String mediaId, String title, String description) {
        String url = UPLOAD_VIDEO_URL + getAccessToken();

        Map<String, String> mapData = new HashMap<>();
        mapData.put("media_id", mediaId);
        mapData.put("title", title);
        mapData.put("description", description);

        String jsonResult = HttpUtils.post(url, JsonUtils.toJson(mapData));
        return ApiResult.build(jsonResult);
    }

    private static final String UPLOAD_NEWS = "https://api.weixin.qq.com/cgi-bin/media/uploadnews?access_token=";

    /**
     * 上传图文消息素材【订阅号与服务号认证后均可用】
     *
     * @param MediaArticle
     * @return {ApiResult}
     */
    public static ApiResult uploadNews(List<MediaArticle> MediaArticle) {
        String url = UPLOAD_NEWS + getAccessToken();

        Map<String, Object> dataMap = new HashMap<>();
        dataMap.put("articles", MediaArticle);

        String jsonResult = HttpUtils.post(url, JsonUtils.toJson(dataMap));
        return ApiResult.build(jsonResult);
    }

    private final static String UPLOAD_IMG_URL = "https://api.weixin.qq.com/cgi-bin/media/uploadimg?access_token=";

    /**
     * 上传图文消息内的图片获取URL
     * 请注意，本接口所上传的图片不占用公众号的素材库中图片数量的5000个的限制。
     * 图片仅支持jpg/png格式，大小必须在1MB以下。
     *
     * @param imgFile 图片文件
     * @return ApiResult
     */
    public static ApiResult uploadImg(File imgFile) {
        String url = UPLOAD_IMG_URL + getAccessToken();

        String jsonResult = HttpUtils.upload(url, imgFile, null);
        return ApiResult.build(jsonResult);
    }

    /**
     * 新增视频永久素材
     * 素材的格式大小等要求与公众平台官网一致。
     * 具体是，图片大小不超过2M，支持bmp/png/jpeg/jpg/gif格式，语音大小不超过5M，长度不超过60秒，支持mp3/wma/wav/amr格式
     *
     * @param file         文件
     * @param title        文件标题
     * @param introduction 介绍
     * @return ApiResult
     */
    public static ApiResult addMaterial(File file, String title, String introduction) {
        String url = ADD_MATERIAL_URL + getAccessToken();

        Map<String, String> dataMap = new HashMap<String, String>();
        dataMap.put("title", title);
        dataMap.put("introduction", introduction);

        String jsonResult = HttpUtils.upload(url, file, JsonUtils.toJson(dataMap));
        return ApiResult.build(jsonResult);
    }

    // 获取永久素材
    private static final String GET_MATERIAL_URL = "https://api.weixin.qq.com/cgi-bin/material/get_material?access_token=";

    /**
     * 获取永久素材
     *
     * @param media_id 要获取的素材的media_id
     * @return InputStream 流，考虑到这里可能返回json或file请自行使用IOUtils转换
     */
    public static InputStream getMaterial(String media_id) {
        String url = GET_MATERIAL_URL + getAccessToken();

        Map<String, Object> dataMap = new HashMap<String, Object>();
        dataMap.put("media_id", media_id);

        return HttpUtils.download(url, JsonUtils.toJson(dataMap));
    }

    // 删除永久素材
    private static final String DEL_MATERIAL_URL = "https://api.weixin.qq.com/cgi-bin/material/del_material?access_token=";

    /**
     * 删除永久素材
     *
     * @param media_id 要获取的素材的media_id
     * @return ApiResult 返回信息
     */
    public static ApiResult delMaterial(String media_id) {
        String url = DEL_MATERIAL_URL + getAccessToken();

        Map<String, Object> dataMap = new HashMap<String, Object>();
        dataMap.put("media_id", media_id);

        String jsonResult = HttpUtils.post(url, JsonUtils.toJson(dataMap));
        return ApiResult.build(jsonResult);
    }

    private static final String UPDATE_NEWS_URL = "https://api.weixin.qq.com/cgi-bin/material/update_news?access_token=";

    /**
     * 修改永久图文素材
     *
     * @param media_id     要修改的图文消息的id
     * @param index        要更新的文章在图文消息中的位置（多图文消息时，此字段才有意义），第一篇为0
     * @param MediaArticle 图文素材
     * @return ApiResult 返回信息
     */
    public static ApiResult updateNews(String media_id, int index, MediaArticle MediaArticle) {
        String url = UPDATE_NEWS_URL + getAccessToken();

        Map<String, Object> dataMap = new HashMap<String, Object>();
        dataMap.put("media_id", media_id);
        dataMap.put("index", index);
        dataMap.put("articles", MediaArticle);

        String jsonResult = HttpUtils.post(url, JsonUtils.toJson(dataMap));
        return ApiResult.build(jsonResult);
    }

    // 获取素材总数
    private static final String GET_MATERIALCOUNT_URL = "https://api.weixin.qq.com/cgi-bin/material/get_materialcount?access_token=";

    /**
     * 获取素材总数
     *
     * @return ApiResult 返回信息
     */
    public static ApiResult getMaterialCount() {
        String url = GET_MATERIALCOUNT_URL + getAccessToken();
        String jsonResult = HttpUtils.get(url);
        return ApiResult.build(jsonResult);
    }

    // 获取素材列表
    private static final String BATCHGET_MATERIAL_URL = "https://api.weixin.qq.com/cgi-bin/material/batchget_material?access_token=";

    /**
     * 获取素材列表
     *
     * @param mediaType 素材的类型，图片（image）、视频（video）、语音 （voice）、图文（news）
     * @param offset    从全部素材的该偏移位置开始返回，0表示从第一个素材 返回
     * @param count     返回素材的数量，取值在1到20之间
     * @return ApiResult 返回信息
     */
    public static ApiResult batchGetMaterial(MediaType mediaType, int offset, int count) {
        String url = BATCHGET_MATERIAL_URL + getAccessToken();

        if (offset < 0) offset = 0;
        if (count > 20) count = 20;
        if (count < 1) count = 1;

        Map<String, Object> dataMap = new HashMap<String, Object>();
        dataMap.put("type", mediaType.getValue());
        dataMap.put("offset", offset);
        dataMap.put("count", count);

        String jsonResult = HttpUtils.post(url, JsonUtils.toJson(dataMap));
        return ApiResult.build(jsonResult);
    }

}
