package com.rocoinfo.weixin.api;

import com.rocoinfo.weixin.model.ApiResult;
import com.rocoinfo.weixin.util.HttpUtils;
import com.rocoinfo.weixin.util.JsonUtils;
import com.rocoinfo.weixin.util.StringUtils;

import java.util.HashMap;
import java.util.Map;

/**
 * <dl>
 * <dd>Description: </dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/4/5 上午11:06</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public class QrcodeApi extends BaseApi {

    private QrcodeApi() {
        super();
    }

    /**
     * 生成二维码的url
     */
    private static final String QRCODE_URL = "https://api.weixin.qq.com/cgi-bin/qrcode/create?access_token=";

    /**
     * 生成带参数的二维码:
     * 参考：https://mp.weixin.qq.com/wiki -> 账号管理 -> 生成带参数的二维码
     *
     * @param json json参数
     * @return
     */
    public static ApiResult create(String json) {
        String url = QRCODE_URL + getAccessToken();
        return ApiResult.build(HttpUtils.post(url, json));
    }

    /**
     * 创建临时二维码
     *
     * @param expireSeconds 该二维码有效时间，以秒为单位。 最大不超过604800（即7天）。
     * @param sceneId       场景值ID，临时二维码时为32位非0整型
     * @return ApiResult 二维码信息
     */
    public static ApiResult createTemporary(long expireSeconds, long sceneId) {
        Map<String, Object> params = new HashMap<>();
        params.put("expire_seconds", expireSeconds);
        params.put("action_name", "QR_SCENE");

        Map<String, Object> actionInfo = new HashMap<>();
        Map<String, Object> scene = new HashMap<>();
        scene.put("scene_id", sceneId);

        actionInfo.put("scene", scene);
        params.put("action_info", actionInfo);
        return create(JsonUtils.toJson(params));
    }

    /**
     * 创建永久二维码
     *
     * @param sceneId 场景值ID，永久二维码时最大值为100000（目前参数只支持1--100000）
     * @return ApiResult 二维码信息
     */
    public static ApiResult createPermanent(long sceneId) {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("action_name", "QR_LIMIT_SCENE");

        Map<String, Object> actionInfo = new HashMap<String, Object>();
        Map<String, Object> scene = new HashMap<String, Object>();
        scene.put("scene_id", sceneId);

        actionInfo.put("scene", scene);
        params.put("action_info", actionInfo);
        return create(JsonUtils.toJson(params));
    }

    /**
     * 创建永久二维码
     *
     * @param sceneStr 场景值ID（字符串形式的ID），字符串类型，长度限制为1到64，仅永久二维码支持此字段
     * @return ApiResult 二维码信息
     */
    public static ApiResult createPermanent(String sceneStr) {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("action_name", "QR_LIMIT_STR_SCENE");

        Map<String, Object> actionInfo = new HashMap<String, Object>();
        Map<String, Object> scene = new HashMap<String, Object>();
        scene.put("scene_str", sceneStr);

        actionInfo.put("scene", scene);
        params.put("action_info", actionInfo);
        return create(JsonUtils.toJson(params));
    }

    private static String SHOW_QRCODE_URL = "https://mp.weixin.qq.com/cgi-bin/showqrcode?ticket=";

    /**
     * 通过ticket换取二维码地址
     *
     * @param ticket 换取二维码参数
     * @return String url
     */
    public static String showQrcodeUrl(String ticket) {
        if (StringUtils.isBlank(ticket))
            return StringUtils.BLANK;
        return SHOW_QRCODE_URL + ticket;
    }

}
