package com.rocoinfo.weixin.token;

import com.rocoinfo.weixin.api.AccessTokenApi;
import com.rocoinfo.weixin.util.StringUtils;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <dl>
 * <dd>Description: 默认提供基于内存的access_token的缓存策略，高并发或集群环境下勿用</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/3/6 下午2:17</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public class DefaultTokenManager implements TokenCacheManager<DefaultTokenCache> {

    private static final String CACHE_KEY = "token_cache_key";

    /**
     * 利用concurrent map来作缓存容器
     */
    private final Map<String, DefaultTokenCache> container = new ConcurrentHashMap<>();

    @Override
    public DefaultTokenCache get() {
        DefaultTokenCache token = container.get(CACHE_KEY);
        if (expire(token)) {
            synchronized (DefaultTokenManager.class) {
                token = container.get(CACHE_KEY);
                if (expire(token)) { // 为了防止并发问题，进行双重判断
                    // 已过期或者缓存中不存在，通过api接口获取access_token
                    String tokenString = AccessTokenApi.getString();
                    if (StringUtils.isNotBlank(tokenString)) {
                        token = new DefaultTokenCache(tokenString);
                        token.setCreateTime(System.currentTimeMillis());
                        container.put(CACHE_KEY, token);
                        return token;
                    }
                }
                return token;
            }
        }
        return token;
    }

    /**
     * 判断access token是否过期,过期返回true，未过期返回false
     *
     * @param token access token
     * @return
     */
    private boolean expire(DefaultTokenCache token) {
        if (token != null && (token.getCreateTime() + token.getExpires() * 1000) > System.currentTimeMillis()) {
            return false;
        }
        return true;
    }

    @Override
    public void clear() {
        container.clear();
    }
}
