package com.rocoinfo.weixin.api;

import com.rocoinfo.weixin.model.payment.PaymentCode;
import com.rocoinfo.weixin.model.payment.req.CloseOrderReq;
import com.rocoinfo.weixin.model.payment.req.DownloadBillReq;
import com.rocoinfo.weixin.model.payment.req.PaymentBaseReq;
import com.rocoinfo.weixin.model.payment.req.UnifiedOrderReq;
import com.rocoinfo.weixin.model.payment.resp.CloseOrderResp;
import com.rocoinfo.weixin.model.payment.resp.DownloadBillResp;
import com.rocoinfo.weixin.model.payment.resp.UnifiedOrderResp;
import com.rocoinfo.weixin.util.HttpUtils;
import com.rocoinfo.weixin.util.PaymentUtils;
import com.rocoinfo.weixin.util.XmlUtils;

/**
 * <dl>
 * <dd>Description: 微信支付工具类</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/9/18 上午11:54</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public class PaymentApi extends BaseApi {

    private PaymentApi() {

    }

    /**
     * 生成指定位数的随机串
     *
     * @param len 随机串位数
     * @return 随机串
     */
    public static String generateNonceStr(int len) {
        return PaymentUtils.generateNonceStr(len);
    }

    /**
     * 根据微信支付统一下单接口的请求数据，生成相应的签名
     *
     * @param req 请求数据
     * @return
     */
    public static <T extends PaymentBaseReq> String calculateSign(T req, String key) {
        return PaymentUtils.calculateSign(req, key);
    }

    /**
     * 统一下单的url
     */
    private static final String UNIFIED_ORDER_URL = "https://api.mch.weixin.qq.com/pay/unifiedorder";

    /**
     * 微信统一下单接口
     *
     * @param req 统一下单接口请求参数
     * @return
     */
    public static UnifiedOrderResp unifiedOrder(UnifiedOrderReq req) {
        if (req == null) {
            throw new RuntimeException("统一下单参数对象为空！");
        }
        String res = HttpUtils.post(UNIFIED_ORDER_URL, XmlUtils.toXml(req));
        return XmlUtils.fromXml(res, UnifiedOrderResp.class);
    }

    private static final String CLOSE_ORDER_URL = "https://api.mch.weixin.qq.com/pay/closeorder";

    /**
     * 微信关闭订单接口
     *
     * @param req 关闭订单接口请求参数
     * @return
     */
    public static CloseOrderResp closeOrder(CloseOrderReq req) {
        if (req == null) {
            throw new RuntimeException("关闭订单参数对象为空！");
        }
        String res = HttpUtils.post(CLOSE_ORDER_URL, XmlUtils.toXml(req));
        return XmlUtils.fromXml(res, CloseOrderResp.class);
    }

    private static final String DOWNLOAD_BILL_URL = "https://api.mch.weixin.qq.com/pay/downloadbill";

    /**
     * 下载对账单接口
     *
     * @param req 下载对账单参数
     * @return
     */
    public static DownloadBillResp downloadBill(DownloadBillReq req) {
        if (req == null) {
            throw new RuntimeException("下载对账单参数对象为空！");
        }
        String res = HttpUtils.post(DOWNLOAD_BILL_URL, XmlUtils.toXml(req));
        if (res.indexOf("<") == 0) {
            return XmlUtils.fromXml(res, DownloadBillResp.class);
        }
        return new DownloadBillResp(PaymentCode.SUCCESS_CODE, "", res);
    }
}
