package com.rocoinfo.weixin.api;

import com.rocoinfo.weixin.model.ApiResult;
import com.rocoinfo.weixin.util.HttpUtils;
import com.rocoinfo.weixin.util.JsonUtils;
import com.rocoinfo.weixin.util.MapUtils;

import java.io.Serializable;
import java.util.Collection;
import java.util.Map;

/**
 * <dl>
 * <dd>Description: 用户标签管理</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/4/7 下午3:19</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public class TagApi extends BaseApi {

    private TagApi() {
    }

    private static final String CREATE_URL = "https://api.weixin.qq.com/cgi-bin/tags/create?access_token=";

    /**
     * 创建标签
     *
     * @param name 标签名称
     * @return
     */
    public static ApiResult create(String name) {
        String url = CREATE_URL + getAccessToken();
        Map<String, String> data = MapUtils.of("name", name);
        Map<String, Map<String, String>> params = MapUtils.of("tag", data);
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }

    private static final String GET_URL = "https://api.weixin.qq.com/cgi-bin/tags/get?access_token=";

    /**
     * 获取公众号已经创建的标签
     *
     * @return
     */
    public static ApiResult getAll() {
        String url = GET_URL + getAccessToken();
        return ApiResult.build(HttpUtils.get(url));
    }

    private static final String EDIT_URL = "https://api.weixin.qq.com/cgi-bin/tags/update?access_token=";

    /**
     * 编辑标签
     *
     * @param id   标签id(微信端id)
     * @param name 标签名称
     * @return
     */
    public static ApiResult edit(long id, String name) {
        String url = EDIT_URL + getAccessToken();
        Map<String, ? extends Serializable> data = MapUtils.of("id", id, "name", name);
        Map<String, ? extends Map<String, ? extends Serializable>> params = MapUtils.of("tag", data);
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }

    private static final String DELETE_URL = "https://api.weixin.qq.com/cgi-bin/tags/delete?access_token=";

    /**
     * 删除标签
     *
     * @param id 标签id(微信端id)
     * @return
     */
    public static ApiResult delete(long id) {
        String url = DELETE_URL + getAccessToken();
        Map<String, Long> data = MapUtils.of("id", id);
        Map<String, Map<String, Long>> params = MapUtils.of("tag", data);
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }

    private static final String GET_FANS_URL = "https://api.weixin.qq.com/cgi-bin/user/tag/get?access_token=";

    /**
     * 获取标签下粉丝列表
     *
     * @param id         标签id（微信端）
     * @param nextOpenid 第一个拉取的OPENID，不填默认从头开始拉取
     * @return
     */
    public static ApiResult getFans(long id, String nextOpenid) {
        String url = GET_FANS_URL + getAccessToken();
        Map<String, ? extends Serializable> data = MapUtils.of("id", id, "next_openid", nextOpenid);
        Map<String, ? extends Map<String, ? extends Serializable>> params = MapUtils.of("tag", data);
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }

    private static final String BATCH_TAG_URL = "https://api.weixin.qq.com/cgi-bin/tags/members/batchtagging?access_token=";

    /**
     * 批量为用户打标签
     *
     * @param id         标签id（微信端）
     * @param collection 用户的openid集合
     * @return
     */
    public static ApiResult batchTag(long id, Collection<String> collection) {
        String url = BATCH_TAG_URL + getAccessToken();
        Map<String, Object> data = MapUtils.of("openid_list", collection, "tagid", id);
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(data)));
    }

    private static final String BATCH_UNTAG_URL = "https://api.weixin.qq.com/cgi-bin/tags/members/batchuntagging?access_token=";

    /**
     * 批量为用户取消标签
     *
     * @param id         标签id（微信端）
     * @param collection 用户的openid集合
     * @return
     */
    public static ApiResult batchUntag(long id, Collection<String> collection) {
        String url = BATCH_UNTAG_URL + getAccessToken();
        Map<String, Object> data = MapUtils.of("openid_list", collection, "tagid", id);
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(data)));
    }

    /**
     * 获取用户身上的标签
     */
    private static final String GET_USER_TAGS_URL = "https://api.weixin.qq.com/cgi-bin/tags/getidlist?access_token=";

    public static ApiResult getUserTags(String openid) {
        String url = GET_USER_TAGS_URL + getAccessToken();
        Map<String, String> params = MapUtils.of("openid", openid);
        return ApiResult.build(HttpUtils.post(url, JsonUtils.toJson(params)));
    }
}
