package com.rocoinfo.weixin.ticket;

import com.rocoinfo.weixin.api.JsApiTicketApi;
import com.rocoinfo.weixin.token.TokenManager;
import com.rocoinfo.weixin.util.StringUtils;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <dl>
 * <dd>Description: 基于内存的jsApiTicket的缓存管理 </dd>
 * <dd>@date：2017/8/17 10:50</dd>
 * </dl>
 */
public class DefaultTicketManager implements TicketCacheManager<DefaultTicketCache> {

    private static final String CACHE_KEY = "ticket_cache_key";

    /**
     * 利用concurrent map来作缓存容器
     */
    private final Map<String, DefaultTicketCache> ticketContainer = new ConcurrentHashMap<>();

    /**
     * 获取jsApiTicket
     * 去缓存ticketContainer中获取,没有就去远程获取
     *
     * @return
     */
    @Override
    public DefaultTicketCache get() {

        //1.先获取 access_token
        String accessToken = new TokenManager().getAccessToken();

        DefaultTicketCache ticket = ticketContainer.get(CACHE_KEY);
        if (expire(ticket)) {
            synchronized (DefaultTicketManager.class) {
                ticket = ticketContainer.get(CACHE_KEY);
                if (expire(ticket)) { // 为了防止并发问题，进行双重判断
                    // 已过期或者缓存中不存在，通过api接口获取 jsApiTicket
                    //2.根据 access_token 获取 jsApiTicket
                    String ticketString = JsApiTicketApi.getTicketStr();
                    if (StringUtils.isNotBlank(ticketString)) {
                        ticket = new DefaultTicketCache(ticketString);
                        ticket.setCreateTime(System.currentTimeMillis());
                        ticketContainer.put(CACHE_KEY, ticket);
                        return ticket;
                    }
                }
                return ticket;
            }
        }
        return ticket;
    }

    /**
     * 判断ticket是否过期,过期返回true，未过期返回false
     *
     * @param ticket ticket
     * @return
     */
    private boolean expire(DefaultTicketCache ticket) {
        if (ticket != null && (ticket.getCreateTime() + ticket.getExpires() * 1000) > System.currentTimeMillis()) {
            return false;
        }
        return true;
    }

    @Override
    public void clear() {
        ticketContainer.clear();
    }
}
