package com.rocoinfo.weixin.util;

import com.rocoinfo.weixin.model.MediaFile;
import okhttp3.*;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * <dl>
 * <dd>Description: http请求工具类</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/3/6 10:22</dd>
 * <dd>@author：Kong</dd>
 * </dl>
 */
public class HttpUtils {

    //http请求工具代理对象
    private static final HttpClient httpClient;

    static {
        httpClient = new Ok3HttpClient();
    }

    /**
     * get请求
     *
     * @param url 请求地址
     * @return 返回请求结果
     */
    public static String get(String url) {
        return httpClient.get(url);
    }

    /**
     * 带参数的get请求
     *
     * @param url        请求url
     * @param queryParas 请求参数
     * @return 返回请求结果
     */
    public static String get(String url, Map<String, String> queryParas) {
        return httpClient.get(url, queryParas);
    }

    /**
     * post请求
     *
     * @param url  请求地址
     * @param data 请求数据
     * @return
     */
    public static String post(String url, String data) {
        return httpClient.post(url, data);
    }

    /**
     * 文件上传接口
     *
     * @param url    url
     * @param file   文件
     * @param params 参数
     * @return
     */
    public static String upload(String url, File file, String params) {
        return httpClient.upload(url, file, params);
    }

    /**
     * 文件下载
     *
     * @param url url
     * @return
     */
    public static MediaFile download(String url) {
        return httpClient.download(url);
    }

    /**
     * 文件下载
     *
     * @param url    请求路径
     * @param params 请求参数
     * @return
     */
    public static InputStream download(String url, String params) {
        return httpClient.download(url, params);
    }


    /**
     * http请求接口规范
     */
    private interface HttpClient {

        String get(String url);

        String get(String url, Map<String, String> params);

        String post(String url, String data);

        String upload(String url, File file, String params);

        MediaFile download(String url);

        InputStream download(String url, String params);
    }

    /**
     * 使用Ok3HttpClient实现的HttpClient
     */
    private static class Ok3HttpClient implements HttpClient {
        /**
         * okhttp请求对象
         */
        private OkHttpClient httpClient;

        public Ok3HttpClient() {
            httpClient = new OkHttpClient().newBuilder()
                    .connectTimeout(10, TimeUnit.SECONDS)
                    .writeTimeout(10, TimeUnit.SECONDS)
                    .readTimeout(30, TimeUnit.SECONDS)
                    .build();
        }

        private static final MediaType CONTENT_TYPE_FORM =
                MediaType.parse("application/x-www-form-urlencoded");

        /**
         * okhttp3处理请求的方法
         *
         * @param request 请求对象
         * @return 返回处理结果
         */
        public String execute(Request request) {
            try {
                Response response = httpClient.newCall(request).execute();
                if (!response.isSuccessful()) {
                    throw new RuntimeException("Unexpected code" + response);
                }
                return response.body().string();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }

        }

        @Override
        public String get(String url) {
            Request request = new Request.Builder().url(url).get().build();
            return execute(request);
        }

        @Override
        public String get(String url, Map<String, String> params) {
            HttpUrl.Builder urlBuilder = HttpUrl.parse(url).newBuilder();
            for (Map.Entry<String, String> entry : params.entrySet()) {
                urlBuilder.addQueryParameter(entry.getKey(), entry.getValue());
            }
            HttpUrl httpUrl = urlBuilder.build();
            Request request = new Request.Builder().url(httpUrl).get().build();
            return execute(request);
        }

        @Override
        public String post(String url, String data) {
            RequestBody body = RequestBody.create(CONTENT_TYPE_FORM, data);
            Request request = new Request.Builder()
                    .url(url)
                    .post(body)
                    .build();
            return execute(request);
        }

        @Override
        public String upload(String url, File file, String params) {
            RequestBody fileBody = RequestBody
                    .create(MediaType.parse("application/octet-stream"), file);

            MultipartBody.Builder builder = new MultipartBody.Builder()
                    .setType(MultipartBody.FORM)
                    .addFormDataPart("media", file.getName(), fileBody);

            if (StringUtils.isNotBlank(params)) {
                builder.addFormDataPart("description", params);
            }

            RequestBody requestBody = builder.build();
            Request request = new Request.Builder()
                    .url(url)
                    .post(requestBody)
                    .build();

            return execute(request);
        }

        @Override
        public MediaFile download(String url) {
            Request request = new Request.Builder().url(url).get().build();
            try {
                Response response = httpClient.newCall(request).execute();

                if (!response.isSuccessful()) throw new RuntimeException("Unexpected code " + response);

                ResponseBody body = response.body();
                MediaType mediaType = body.contentType();
                MediaFile mediaFile = new MediaFile();
                if (mediaType.type().equals("text")) {
                    mediaFile.setError(body.string());
                } else {
                    BufferedInputStream bis = new BufferedInputStream(body.byteStream());

                    String ds = response.header("Content-disposition");
                    String fullName = ds.substring(ds.indexOf("filename=\"") + 10, ds.length() - 1);
                    String relName = fullName.substring(0, fullName.lastIndexOf("."));
                    String suffix = fullName.substring(relName.length() + 1);

                    mediaFile.setFullName(fullName);
                    mediaFile.setFileName(relName);
                    mediaFile.setSuffix(suffix);
                    mediaFile.setContentLength(body.contentLength() + "");
                    mediaFile.setContentType(body.contentType().toString());
                    mediaFile.setFileStream(bis);
                }
                return mediaFile;
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public InputStream download(String url, String params) {
            Request request;
            if (StringUtils.isNotBlank(params)) {
                RequestBody body = RequestBody.create(CONTENT_TYPE_FORM, params);
                request = new Request.Builder().url(url).post(body).build();
            } else {
                request = new Request.Builder().url(url).get().build();
            }
            try {
                Response response = httpClient.newCall(request).execute();

                if (!response.isSuccessful()) throw new RuntimeException("Unexpected code " + response);

                return response.body().byteStream();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
