package com.rocoinfo.weixin.api;

import com.rocoinfo.weixin.config.ParamManager;
import com.rocoinfo.weixin.model.ApiResult;
import com.rocoinfo.weixin.util.HttpUtils;
import com.rocoinfo.weixin.util.StringUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

/**
 * <dl>
 * <dd>Description: 用户授权工具类</dd>
 * <dd>Company: 大城若谷信息技术有限公司</dd>
 * <dd>@date：2017/3/15 上午11:13</dd>
 * <dd>@author：Aaron</dd>
 * </dl>
 */
public class OAuthApi {

    private OAuthApi() {
        super();
    }

    /**
     * 用户两种授权方式,base和info
     * base：获取用户openid
     * info：获取用户详细信息
     */
    public enum Type {
        BASE("snsapi_base"), INFO("snsapi_userinfo");

        private String value;

        Type(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    /**
     * oauth授权引导用户点击的url
     */
    private static final String OAUTH_URL = "https://open.weixin.qq.com/connect/oauth2/authorize?appid=%s&redirect_uri=%s&response_type=code&scope=%s&state=%s#wechat_redirect";

    /**
     * 构建用户授权的url（引导用户点击此url才能使用微信oauth授权）
     * 在构建过程中会对url进行encode,默认字符集UTF-8
     *
     * @param url   微信跳转回调的redirect_url
     * @param state 用户自定义的state
     * @param type  授权类型，base或info
     * @return
     */
    public static String buildUrl(String url, String state, Type type) {
        if (StringUtils.isBlank(url) || type == null)
            throw new IllegalArgumentException("url或type为不能为空");
        return String.format(OAUTH_URL, ParamManager.getAppid(), URLEncode(url), type.getValue(), state);
    }

    /**
     * 获取access_token的url
     */
    private static final String GET_ACCESS_TOKEN_URL = "https://api.weixin.qq.com/sns/oauth2/access_token?appid=%s&secret=%s&code=%s&grant_type=authorization_code";

    /**
     * 根据code换区access_token，获取access_token时会获取到用户的openid
     * 注意：这是一个特殊的access_token，只用于网页授权，和作为全局凭证的access_token不同
     *
     * @param code 微信端传递的
     * @return
     */
    public static ApiResult getAccessToken(String code) {
        if (StringUtils.isBlank(code)) {
            throw new NullPointerException("code is null");
        }
        String url = String.format(GET_ACCESS_TOKEN_URL, ParamManager.getAppid(), ParamManager.getSecret(), code);
        return ApiResult.build(HttpUtils.get(url));
    }

    /**
     * 根据code换区access_token，获取access_token时会获取到用户的openid
     * 注意：这是一个特殊的access_token，只用于网页授权，和作为全局凭证的access_token不同
     *
     * @param code 微信端传递的
     * @return
     */
    public static String getOpenid(String code) {
        ApiResult res = getAccessToken(code);
        if (res.isSuccess()) {
            return (String) res.fromJsonAsMap().get("openid");
        }
        return null;
    }

    private static final String REFRESH_ACCESS_TOKEN_URL = "https://api.weixin.qq.com/sns/oauth2/refresh_token?appid=%s&grant_type=refresh_token&refresh_token=%s";

    /**
     * 由于access_token拥有较短的有效期，当access_token超时后，可以使用refresh_token进行刷新，
     * refresh_token拥有较长的有效期（7天、30天、60天、90天），当refresh_token失效的后，需要用户重新授权。
     *
     * @param refreshToken 通过access_token获取到的refresh_token参数
     * @return
     */
    public static ApiResult refreshAccessToken(String refreshToken) {
        if (StringUtils.isBlank(refreshToken)) {
            throw new NullPointerException("refresh token is null");
        }
        String url = String.format(REFRESH_ACCESS_TOKEN_URL, ParamManager.getAppid(), refreshToken);
        return ApiResult.build(HttpUtils.get(url));
    }

    private static final String GET_USER_INFO_URL = "https://api.weixin.qq.com/sns/userinfo?access_token=%s&openid=%s&lang=%s";

    private static final String DEFAULT_LANG = "zh_CN";

    /**
     * 如果网页授权作用域为snsapi_userinfo，则此时开发者可以通过access_token和openid拉取用户信息
     *
     * @param accessTokne access token
     * @param openid      openid
     * @param lang        语言：zh_CN 简体，zh_TW 繁体，en 英语，不传默认为zh_CN
     * @return
     */
    public static ApiResult getUserInfo(String accessTokne, String openid, String lang) {
        if (StringUtils.isAnyBlank(accessTokne, openid)) {
            throw new NullPointerException("access token or openid is null");
        }
        if (StringUtils.isBlank(lang))
            lang = DEFAULT_LANG;
        String url = String.format(GET_USER_INFO_URL, accessTokne, lang);
        return ApiResult.build(HttpUtils.get(url));
    }


    /**
     * 默认url encode字符集
     */
    private final static String DEFAULT_ENCODE_CHARACTER = "UTF-8";

    /**
     * 对传入的字符串进行编码，默认字符集utf-8
     *
     * @param s string
     * @return
     */
    private static String URLEncode(String s) {
        String res = null;
        try {
            res = URLEncoder.encode(s, DEFAULT_ENCODE_CHARACTER);
        } catch (UnsupportedEncodingException e) {
            // 编码失败直接抛runtime异常
            throw new RuntimeException("url encoding failed!");
        }
        return res;
    }
}
